<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Models\DeliveryAgent;
use App\Models\User;

class DeliveryAgentSettingsController extends Controller
{
    /**
     * Get current settings/status.
     */
    public function index(Request $request)
    {
        $user = $request->user();
        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();

        return response()->json([
            'is_available' => (bool) $agent->is_available,
            'is_active' => (bool) $user->is_active,
        ]);
    }

    /**
     * Toggle availability status.
     */
    public function toggleAvailability(Request $request)
    {
        $user = $request->user();
        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();

        $agent->is_available = !$agent->is_available;
        $agent->save();

        return response()->json([
            'message' => 'Availability updated successfully',
            'is_available' => (bool) $agent->is_available,
        ]);
    }

    /**
     * Update password.
     */
    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'password' => 'required|min:8|confirmed',
        ]);

        $user = $request->user();

        if (!Hash::check($request->current_password, $user->password)) {
            return response()->json([
                'message' => 'The provided current password does not match our records.'
            ], 422);
        }

        $user->update([
            'password' => Hash::make($request->password),
        ]);

        return response()->json([
            'message' => 'Password updated successfully'
        ]);
    }

    /**
     * Deactivate account.
     */
    public function deactivateAccount(Request $request)
    {
        $user = $request->user();

        // Mark user as inactive
        $user->update(['is_active' => false]);

        // Sync with DeliveryAgent profile status
        $agent = DeliveryAgent::where('user_id', $user->id)->first();
        if ($agent) {
            $agent->update(['status' => false, 'is_available' => false]);
        }

        // Revoke all tokens
        $user->tokens()->delete();

        return response()->json([
            'message' => 'Account deactivated successfully. You have been logged out.'
        ]);
    }
}
