<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Models\DeliveryAgent;
use App\Models\User;

class DeliveryAgentProfileController extends Controller
{
    /**
     * Get the delivery agent's profile.
     */
    public function show(Request $request)
    {
        $user = $request->user();
        $agent = DeliveryAgent::where('user_id', $user->id)
            ->with(['documents', 'vehicles'])
            ->first();

        if (!$agent) {
            return response()->json(['message' => 'Delivery agent profile not found'], 404);
        }

        return response()->json([
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'profile_photo' => $user->profile_photo_path ? asset('storage/' . $user->profile_photo_path) : null,
            ],
            'agent' => [
                'id' => $agent->id,
                'phone' => $agent->phone,
                'alternate_phone' => $agent->alternate_phone,
                'dob' => $agent->dob instanceof \Carbon\Carbon ? $agent->dob->format('Y-m-d') : ($agent->dob ? date('Y-m-d', strtotime($agent->dob)) : null),
                'gender' => $agent->gender,
                'address' => $agent->address,
                'is_available' => $agent->is_available,
                'bank_name' => $agent->bank_name,
                'account_number' => $agent->account_number,
                'ifsc_code' => $agent->ifsc_code,
            ],
            'vehicles' => $agent->vehicles,
            'documents' => $agent->documents->map(function ($doc) {
                return [
                    'id' => $doc->id,
                    'type' => $doc->type,
                    'number' => $doc->number,
                    'image' => asset('storage/' . $doc->image_path),
                ];
            }),
        ]);
    }

    /**
     * Update the delivery agent's profile.
     */
    public function update(Request $request)
    {
        $user = $request->user();
        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();

        $request->validate([
            'name' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
            'alternate_phone' => 'nullable|string|max:20',
            'dob' => 'nullable|date',
            'gender' => 'nullable|in:Male,Female,Other',
            'address' => 'nullable|string',
            'profile_photo' => 'nullable|image|max:2048',
            'bank_name' => 'nullable|string|max:255',
            'account_number' => 'nullable|string|max:255',
            'ifsc_code' => 'nullable|string|max:255',
        ]);

        // Update User fields
        if ($request->has('name')) {
            $user->name = $request->name;
        }

        if ($request->hasFile('profile_photo')) {
            if ($user->profile_photo_path) {
                Storage::disk('public')->delete($user->profile_photo_path);
            }
            $user->profile_photo_path = $request->file('profile_photo')->store('profile-photos', 'public');
        }
        $user->save();

        // Update Agent fields
        $agent->update($request->only([
            'phone',
            'alternate_phone',
            'dob',
            'dob',
            'gender',
            'address',
            'bank_name',
            'account_number',
            'ifsc_code'
        ]));

        return response()->json([
            'message' => 'Profile updated successfully',
            'user' => [
                'name' => $user->name,
                'profile_photo' => $user->profile_photo_path ? asset('storage/' . $user->profile_photo_path) : null,
            ],
            'agent' => $agent->only(['phone', 'alternate_phone', 'dob', 'gender', 'address', 'bank_name', 'account_number', 'ifsc_code'])
        ]);
    }
    /**
     * Update the FCM token for push notifications.
     */
    public function updateFcmToken(Request $request)
    {
        $request->validate([
            'fcm_token' => 'required|string',
        ]);

        $user = $request->user();
        $user->update([
            'fcm_token' => $request->fcm_token,
        ]);

        return response()->json([
            'message' => 'FCM token updated successfully',
        ]);
    }
}
