<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Order;
use App\Models\DeliveryAgent;

class DeliveryAgentOrderController extends Controller
{
    /**
     * List all orders for the agent with filters.
     */
    public function index(Request $request)
    {
        $user = $request->user();
        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();
        $status = $request->get('status', 'assigned'); // assigned, transit, completed, cancelled

        $query = Order::where('delivery_agent_id', $agent->id);

        if ($status === 'assigned') {
            $query->whereIn('status', ['pending', 'processing']);
        } elseif ($status === 'transit') {
            $query->where('status', 'shipped');
        } elseif ($status === 'completed') {
            $query->where('status', 'completed');
        } elseif ($status === 'cancelled') {
            $query->where('status', 'cancelled');
        }

        $orders = $query->with(['pickupStore', 'user', 'shippingAddress'])
            ->latest()
            ->paginate(15);

        // Calculate counts for filters
        $counts = [
            'assigned_count' => Order::where('delivery_agent_id', $agent->id)->whereIn('status', ['pending', 'processing'])->count(),
            'transit_count' => Order::where('delivery_agent_id', $agent->id)->where('status', 'shipped')->count(),
            'completed_count' => Order::where('delivery_agent_id', $agent->id)->where('status', 'completed')->count(),
            'cancelled_count' => Order::where('delivery_agent_id', $agent->id)->where('status', 'cancelled')->count(),
        ];

        return response()->json([
            'orders' => $orders,
            'counts' => $counts
        ]);
    }

    /**
     * Get detailed order info.
     */
    public function show(Request $request, $id)
    {
        $user = $request->user();
        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();
        $order = Order::with(['items.product', 'items.variation', 'shippingAddress', 'billingAddress', 'user', 'pickupStore', 'agentRating', 'productReviews.product'])
            ->where('delivery_agent_id', $agent->id)
            ->where('id', $id)
            ->firstOrFail();

        return response()->json([
            'order' => $order,
            'customer_details' => [
                'name' => $order->user ? $order->user->name : ($order->shippingAddress ? $order->shippingAddress->name : 'N/A'),
                'phone' => $order->user ? $order->user->whatsapp_number ?? $order->user->email : ($order->shippingAddress ? $order->shippingAddress->phone : 'N/A'),
                'address' => $order->shippingAddress ? [
                    'line1' => $order->shippingAddress->address_line1,
                    'line2' => $order->shippingAddress->address_line2,
                    'city' => $order->shippingAddress->city,
                    'state' => $order->shippingAddress->state,
                    'postal_code' => $order->shippingAddress->postal_code,
                    'country' => $order->shippingAddress->country,
                    'maps_url' => "https://www.google.com/maps/search/?api=1&query=" . urlencode($order->shippingAddress->address_line1 . " " . $order->shippingAddress->city)
                ] : null
            ],
            'pickup_location' => $order->pickupStore ? [
                'store_name' => $order->pickupStore->name,
                'contact' => $order->pickupStore->phone ?? 'N/A',
                'address' => $order->pickupStore->address,
                'city' => $order->pickupStore->city,
                'state' => $order->pickupStore->state,
                'get_directions_url' => "https://www.google.com/maps/dir/?api=1&destination=" . urlencode($order->pickupStore->address)
            ] : null,
            'payment_info' => [
                'method' => $order->payment_method,
                'status' => $order->payment_status,
                'total' => $order->total,
                'must_collect_cash' => ($order->payment_method === 'cod' && $order->payment_status === 'pending')
            ],
            'customer_feedback' => ($order->status === 'completed') ? [
                'agent_rating' => $order->agentRating,
                'product_reviews' => $order->productReviews
            ] : null
        ]);
    }

    /**
     * Update order status.
     */
    public function updateStatus(Request $request, $id)
    {
        $user = $request->user();
        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();
        $order = Order::where('delivery_agent_id', $agent->id)->where('id', $id)->firstOrFail();

        $request->validate([
            'status' => 'required|in:shipped,completed,cancelled',
            'reason' => 'required_if:status,cancelled|nullable|string|max:1000',
        ]);

        $order->update([
            'status' => $request->status,
        ]);

        if ($request->status == 'cancelled' && $request->filled('reason')) {
            $order->update(['cancellation_reason' => $request->reason]);
        }

        // Logic for completion (COD Payment & Wallet)
        if ($request->status == 'completed') {
            $orderUser = $order->user;
            if ($orderUser) {
                // Mark as paid if COD
                if ($order->payment_method == 'cod' && $order->payment_status == 'pending') {
                    $order->update(['payment_status' => 'paid']);
                }

                // Wallet rewards logic can be triggered here similar to web/admin controllers
                $settings = \App\Models\GeneralSetting::firstOrNew();
                if ($settings->wallet_enabled && $order->payment_status != 'refunded') {
                    $alreadyEarned = $orderUser->walletTransactions()
                        ->where('reference_type', 'App\Models\Order')
                        ->where('reference_id', $order->id)
                        ->where('type', 'credit')
                        ->where('description', 'like', 'Earned%')
                        ->exists();

                    if (!$alreadyEarned && $settings->wallet_earn_rate > 0) {
                        $earnPoints = floor(($order->total / 100) * $settings->wallet_earn_rate);
                        if ($earnPoints > 0) {
                            $orderUser->creditWallet($earnPoints, "Earned from Order #" . $order->order_number, 'App\Models\Order', $order->id);
                        }
                    }
                }
            }
        }

        return response()->json([
            'message' => 'Order status updated to ' . $request->status,
            'order_status' => $order->status
        ]);
    }
    /**
     * Get dashboard statistics.
     */
    public function dashboard(Request $request)
    {
        $user = $request->user();
        if (!$user->hasRole('delivery_agent')) {
            return response()->json(['message' => 'Access denied'], 403);
        }

        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();

        // Stats
        $newOrders = Order::where('delivery_agent_id', $agent->id)->whereIn('status', ['pending', 'processing'])->count();
        $inTransit = Order::where('delivery_agent_id', $agent->id)->where('status', 'shipped')->count();
        $completed = Order::where('delivery_agent_id', $agent->id)->where('status', 'completed')->count();
        $cancelled = Order::where('delivery_agent_id', $agent->id)->where('status', 'cancelled')->count();

        $cashCollected = Order::where('delivery_agent_id', $agent->id)
            ->where('status', 'completed')
            ->where('payment_method', 'cod')
            ->sum('total');

        $codOrderCount = Order::where('delivery_agent_id', $agent->id)
            ->where('status', 'completed')
            ->where('payment_method', 'cod')
            ->count();

        $unreadNotifications = $user->unreadNotifications->count();

        return response()->json([
            'agent' => [
                'name' => $user->name,
                'is_available' => $agent->is_available,
                'average_rating' => round($agent->average_rating, 1),
                'wallet_balance' => $user->wallet_balance,
                'fcm_token' => $user->fcm_token
            ],
            'stats' => [
                'new_orders' => $newOrders,
                'in_transit' => $inTransit,
                'completed' => $completed,
                'cancelled' => $cancelled,
                'cash_collected' => $cashCollected,
                'cod_order_count' => $codOrderCount,
                'unread_notifications' => $unreadNotifications
            ]
        ]);
    }
}
