<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Wishlist;
use App\Models\Product;

class CustomerWishlistController extends Controller
{
    /**
     * List all items in the customer's wishlist.
     */
    public function index(Request $request)
    {
        $wishlist = $request->user()->wishlist()
            ->with(['product.images', 'product.category'])
            ->latest()
            ->paginate(15);

        return response()->json($wishlist);
    }

    /**
     * Toggle a product in/out of the wishlist.
     */
    public function toggle(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
        ]);

        $userId = $request->user()->id;
        $productId = $request->product_id;

        $wishlistItem = Wishlist::where('user_id', $userId)
            ->where('product_id', $productId)
            ->first();

        if ($wishlistItem) {
            $wishlistItem->delete();
            return response()->json([
                'status' => 'removed',
                'message' => 'Product removed from wishlist.'
            ]);
        } else {
            Wishlist::create([
                'user_id' => $userId,
                'product_id' => $productId,
            ]);
            return response()->json([
                'status' => 'added',
                'message' => 'Product added to wishlist.'
            ], 201);
        }
    }

    /**
     * Remove a specific item from the wishlist.
     */
    public function destroy(Request $request, $productId)
    {
        $deleted = Wishlist::where('user_id', $request->user()->id)
            ->where('product_id', $productId)
            ->delete();

        if ($deleted) {
            return response()->json(['message' => 'Product removed from wishlist.']);
        }

        return response()->json(['message' => 'Product not found in wishlist.'], 404);
    }

    /**
     * Clear the entire wishlist.
     */
    public function clear(Request $request)
    {
        $request->user()->wishlist()->delete();

        return response()->json(['message' => 'Wishlist cleared successfully.']);
    }
}
