<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Ticket;
use App\Models\TicketMessage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class CustomerTicketController extends Controller
{
    /**
     * List my support tickets.
     */
    public function index(Request $request)
    {
        $tickets = $request->user()->tickets()
            ->with(['order:id,order_number'])
            ->latest()
            ->paginate(15);

        return response()->json($tickets);
    }

    /**
     * Create a new support ticket.
     */
    public function store(Request $request)
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id',
            'subject' => 'required|string|max:255',
            'message' => 'required|string',
            'attachment' => 'nullable|file|max:5120', // Max 5MB
        ]);

        DB::beginTransaction();

        try {
            // Generate unique ticket number
            $ticketNumber = 'TKT-' . strtoupper(Str::random(8));
            while (Ticket::where('ticket_number', $ticketNumber)->exists()) {
                $ticketNumber = 'TKT-' . strtoupper(Str::random(8));
            }

            $ticket = Ticket::create([
                'user_id' => $request->user()->id,
                'order_id' => $request->order_id,
                'ticket_number' => $ticketNumber,
                'subject' => $request->subject,
                'status' => 'open',
            ]);

            $attachmentPath = null;
            if ($request->hasFile('attachment')) {
                $attachmentPath = $request->file('attachment')->store('tickets/attachments', 'public');
            }

            $ticket->messages()->create([
                'user_id' => $request->user()->id,
                'message' => $request->message,
                'attachment_path' => $attachmentPath,
            ]);

            DB::commit();

            return response()->json([
                'message' => 'Support ticket created successfully.',
                'ticket' => $ticket->load('messages')
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Failed to create ticket: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Get ticket details with message history.
     */
    public function show(Request $request, $id)
    {
        $ticket = $request->user()->tickets()
            ->with(['order', 'messages.user:id,name,profile_photo_path'])
            ->findOrFail($id);

        return response()->json($ticket);
    }

    /**
     * Reply to a support ticket.
     */
    public function reply(Request $request, $id)
    {
        $ticket = $request->user()->tickets()->findOrFail($id);

        if ($ticket->status === 'closed') {
            return response()->json(['message' => 'Cannot reply to a closed ticket.'], 403);
        }

        $request->validate([
            'message' => 'required|string',
            'attachment' => 'nullable|file|max:5120',
        ]);

        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $attachmentPath = $request->file('attachment')->store('tickets/attachments', 'public');
        }

        $message = $ticket->messages()->create([
            'user_id' => $request->user()->id,
            'message' => $request->message,
            'attachment_path' => $attachmentPath,
        ]);

        return response()->json([
            'message' => 'Reply sent successfully.',
            'ticket_message' => $message
        ], 201);
    }

    /**
     * Mark a ticket as closed.
     */
    public function close(Request $request, $id)
    {
        $ticket = $request->user()->tickets()->findOrFail($id);

        if ($ticket->status === 'closed') {
            return response()->json(['message' => 'Ticket is already closed.'], 400);
        }

        $ticket->update(['status' => 'closed']);

        return response()->json([
            'message' => 'Ticket closed successfully.',
            'ticket' => $ticket
        ]);
    }
}
