<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class CustomerSettingsController extends Controller
{
    /**
     * Update customer profile details.
     */
    public function updateProfile(Request $request)
    {
        $user = $request->user();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'gender' => ['nullable', Rule::in(['Male', 'Female', 'Other'])],
            'dob' => 'nullable|date',
            'whatsapp_number' => 'nullable|string|max:20',
            'profile_image' => 'nullable|image|max:2048', // Max 2MB
        ]);

        if ($request->hasFile('profile_image')) {
            // Delete old image if exists
            if ($user->profile_photo_path && Storage::disk('public')->exists($user->profile_photo_path)) {
                Storage::disk('public')->delete($user->profile_photo_path);
            }

            $path = $request->file('profile_image')->store('profile_images', 'public');
            $validated['profile_photo_path'] = $path;
        }

        $user->update($validated);

        return response()->json([
            'message' => 'Profile updated successfully.',
            'user' => $user
        ]);
    }

    /**
     * Update account password.
     */
    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'password' => 'required|string|min:8|confirmed',
        ]);

        $user = $request->user();

        if (!Hash::check($request->current_password, $user->password)) {
            return response()->json([
                'message' => 'The provided current password does not match our records.'
            ], 422);
        }

        $user->update([
            'password' => Hash::make($request->password),
        ]);

        return response()->json([
            'message' => 'Password updated successfully.'
        ]);
    }

    /**
     * Deactivate the customer account.
     */
    public function deactivateAccount(Request $request)
    {
        $user = $request->user();

        // Mark user as inactive or deactivated based on table structure
        // Looking at User model, it has 'is_active' and 'is_deactivated'
        $user->update([
            'is_active' => false,
            'is_deactivated' => true
        ]);

        // Revoke all tokens to log out everywhere
        $user->tokens()->delete();

        return response()->json([
            'message' => 'Your account has been deactivated. You have been logged out.'
        ]);
    }

    /**
     * Get current user settings/profile summary.
     */
    public function index(Request $request)
    {
        $user = $request->user();

        return response()->json([
            'user' => $user,
            'settings' => [
                'is_active' => (bool) $user->is_active,
                'notifications_enabled' => true, // Placeholder if you add push settings later
            ]
        ]);
    }

    /**
     * Update FCM Token for push notifications.
     */
    public function updateFcmToken(Request $request)
    {
        $request->validate([
            'fcm_token' => 'required|string',
        ]);

        $user = $request->user();

        // Update legacy column
        $user->update([
            'fcm_token' => $request->fcm_token
        ]);

        // Store in JSON file (to avoid DB migration)
        try {
            $path = "fcm_tokens/{$user->id}.json";
            $tokens = [];

            if (Storage::exists($path)) {
                $content = Storage::get($path);
                $tokens = json_decode($content, true) ?? [];
            }

            if (!in_array($request->fcm_token, $tokens)) {
                $tokens[] = $request->fcm_token;
                // Optional: Limit number of tokens per user to avoid bloat (e.g., max 10)
                if (count($tokens) > 10) {
                    array_shift($tokens); // Remove oldest
                }
                Storage::put($path, json_encode($tokens));
            }

        } catch (\Exception $e) {
            // Log error but don't fail request
            Log::error("Failed to store FCM token in file: " . $e->getMessage());
        }

        return response()->json([
            'message' => 'FCM Token updated successfully.'
        ]);
    }
}
