<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\GeneralSetting;

class CustomerProductController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::where('status', true);
        $search = $request->input('search');

        if ($search) {
            // Check if search matches a category name
            $category = \App\Models\Category::where('name', 'like', '%' . $search . '%')->first();

            if ($category) {
                // If category found, search products in this category OR matching the name
                $query->where(function ($q) use ($category, $search) {
                    $q->where('category_id', $category->id)
                        ->orWhere('subcategory_id', $category->id)
                        ->orWhere('name', 'like', '%' . $search . '%');
                });
            } else {
                // Determine if searching by plain text or product name
                $query->where('name', 'like', '%' . $search . '%');
            }
        }

        if ($request->has('category_id')) {
            if (is_array($request->category_id)) {
                $query->whereIn('category_id', $request->category_id);
            } else {
                $query->where('category_id', $request->category_id);
            }
        }

        if ($request->has('subcategory_id')) {
            if (is_array($request->subcategory_id)) {
                $query->whereIn('subcategory_id', $request->subcategory_id);
            } else {
                $query->where('subcategory_id', $request->subcategory_id);
            }
        }

        if ($request->has('min_price')) {
            $query->where('sp', '>=', $request->min_price);
        }

        if ($request->has('max_price')) {
            $query->where('sp', '<=', $request->max_price);
        }

        // Clone query for price limits (before applying price filters)
        $priceQuery = clone $query;
        $minPrice = $priceQuery->min('sp') ?? 0;
        $maxPrice = $priceQuery->max('sp') ?? 0;

        if ($request->has('min_price')) {
            $query->where('sp', '>=', $request->min_price);
        }

        if ($request->has('max_price')) {
            $query->where('sp', '<=', $request->max_price);
        }

        // Basic sorting
        if ($request->has('sort')) {
            switch ($request->sort) {
                case 'price_asc':
                    $query->orderBy('sp', 'asc');
                    break;
                case 'price_desc':
                    $query->orderBy('sp', 'desc');
                    break;
                case 'newest':
                    $query->orderBy('created_at', 'desc');
                    break;
                default:
                    $query->orderBy('id', 'desc');
            }
        } else {
            $query->orderBy('id', 'desc');
        }

        $products = $query->with(['images', 'variations.options'])->paginate(20);

        $settings = GeneralSetting::firstOrNew();
        $minStock = $settings->min_stock_threshold ?? 5;

        $products->getCollection()->transform(function ($product) use ($minStock) {
            $firstVariation = $product->variations->first();
            $product->attribute_value = $firstVariation?->options->first()?->attribute_value ?? '';

            // Add stock message
            foreach ($product->variations as $variation) {
                if ($variation->stock <= 0) {
                    $variation->stock_msg = "Out of Stock";
                } elseif ($variation->stock <= $minStock) {
                    $variation->stock_msg = "Only {$variation->stock} left!";
                } else {
                    $variation->stock_msg = "In Stock";
                }
            }

            return $product;
        });

        // Get filter categories (Parent categories)
        $filterCategories = \App\Models\Category::where('status', true)
            ->whereNull('parent_id')
            ->select('id', 'name')
            ->with([
                'children' => function ($q) {
                    $q->where('status', true)->select('id', 'name', 'parent_id');
                }
            ])
            ->get();

        // Custom Response Structure
        return response()->json([
            'data' => $products->items(),
            'filters' => [
                'min_price' => (float) $minPrice,
                'max_price' => (float) $maxPrice,
                'categories' => $filterCategories
            ],
            'meta' => [
                'current_page' => $products->currentPage(),
                'last_page' => $products->lastPage(),
                'per_page' => $products->perPage(),
                'total' => $products->total(),
            ],
            'links' => [
                'first' => $products->url(1),
                'last' => $products->url($products->lastPage()),
                'prev' => $products->previousPageUrl(),
                'next' => $products->nextPageUrl(),
            ]
        ]);
    }

    public function show($id)
    {
        $product = Product::where('status', true)
            ->with(['variations.options', 'images', 'reviews', 'category', 'subcategory'])
            ->findOrFail($id);

        $settings = GeneralSetting::firstOrNew();
        $minStock = $settings->min_stock_threshold ?? 5;

        // Process main product variations
        foreach ($product->variations as $variation) {
            if ($variation->stock <= 0) {
                $variation->stock_msg = "Out of Stock";
            } elseif ($variation->stock <= $minStock) {
                $variation->stock_msg = "Only {$variation->stock} left!";
            } else {
                $variation->stock_msg = "In Stock";
            }
        }

        // Fetch "You may also like" products
        $relatedProductsQuery = Product::where('status', true)
            ->where('id', '!=', $product->id)
            ->with(['images', 'variations.options']);

        if ($product->subcategory_id) {
            $relatedProductsQuery->where('subcategory_id', $product->subcategory_id);
        } else {
            $relatedProductsQuery->where('category_id', $product->category_id);
        }

        $relatedProducts = $relatedProductsQuery->limit(10)->get();

        // If not enough products in subcategory, fill with category products
        if ($relatedProducts->count() < 4 && $product->subcategory_id) {
            $extraProducts = Product::where('status', true)
                ->where('category_id', $product->category_id)
                ->where('id', '!=', $product->id)
                ->whereNotIn('id', $relatedProducts->pluck('id'))
                ->with(['images', 'variations.options'])
                ->limit(10 - $relatedProducts->count())
                ->get();

            $relatedProducts = $relatedProducts->concat($extraProducts);
        }

        $relatedProducts->transform(function ($product) use ($minStock) {
            $firstVariation = $product->variations->first();
            $product->attribute_value = $firstVariation?->options->first()?->attribute_value ?? '';

            // Add stock message
            foreach ($product->variations as $variation) {
                if ($variation->stock <= 0) {
                    $variation->stock_msg = "Out of Stock";
                } elseif ($variation->stock <= $minStock) {
                    $variation->stock_msg = "Only {$variation->stock} left!";
                } else {
                    $variation->stock_msg = "In Stock";
                }
            }

            return $product;
        });

        // Convert to array and add related products
        $data = $product->toArray();
        $data['related_products'] = $relatedProducts;

        return response()->json($data);
    }
    public function getImagesBatch(Request $request)
    {
        $ids = $request->input('ids', []);

        if (is_string($ids)) {
            $ids = explode(',', $ids);
        }

        if (empty($ids)) {
            return response()->json([]);
        }

        $products = Product::whereIn('id', $ids)
            ->with([
                'images' => function ($q) {
                    $q->orderBy('sort_order');
                }
            ])
            ->get(['id']);

        $res = [];
        foreach ($products as $product) {
            $res[$product->id] = $product->images->pluck('image_path');
        }

        return response()->json($res);
    }
}
