<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Order;
use App\Models\ProductReview;
use App\Models\AgentRating;
use App\Models\Cart;
use App\Models\ProductVariation;

class CustomerOrderController extends Controller
{
    /**
     * List customer orders.
     */
    public function index(Request $request)
    {
        $orders = $request->user()->orders()
            ->with(['items.product.images']) // Eager load product images
            ->where('status', '!=', 'pending_payment')
            ->latest()
            ->paginate(15);

        // Transform collection to map 'completed' -> 'delivered'
        $orders->getCollection()->transform(function ($order) {
            if ($order->status === 'completed') {
                $order->status = 'delivered';
            }
            return $order;
        });

        return response()->json($orders);
    }

    /**
     * Get order details.
     */
    public function show(Request $request, $id)
    {
        $order = $request->user()->orders()
            ->with(['items.product.images', 'deliveryAgent.user', 'shippingAddress', 'agentRating', 'productReviews.product', 'statusHistories'])
            ->findOrFail($id);

        $canCallAgent = false;
        $agentPhone = null;

        if ($order->deliveryAgent && $order->order_scope === 'Local' && $order->status === 'shipped') {
            $canCallAgent = true;
            $agentPhone = $order->deliveryAgent->phone;
        }

        // Override status 'completed' -> 'delivered' for API
        $orderArray = $order->toArray();
        if ($orderArray['status'] === 'completed') {
            $orderArray['status'] = 'delivered';
        }

        return response()->json([
            'order' => $orderArray,
            'pricing' => [
                'subtotal' => (float) $order->subtotal,
                'shipping' => (float) $order->shipping_cost,
                'discount' => (float) $order->discount,
                'coupon_code' => $order->coupon_code,
                'points_redeemed' => (int) $order->points_redeemed,
                'points_discount' => (float) $order->points_discount,
                'gift_card_discount' => (float) $order->gift_card_discount,
                'total' => (float) $order->total,
            ],
            'cancellation_reason' => $order->cancellation_reason,
            'timeline' => $order->statusHistories->map(function ($history) {
                // Map 'completed' -> 'delivered'
                $status = $history->status === 'completed' ? 'delivered' : $history->status;
                $statusLabel = ucfirst(str_replace('_', ' ', $status));

                return [
                    'status' => $status,
                    'created_at' => $history->created_at,
                    'description' => 'Order was marked as ' . $statusLabel
                ];
            }),
            'delivery_agent' => $order->deliveryAgent ? [
                'name' => $order->deliveryAgent->user->name,
                'phone' => $order->deliveryAgent->phone,
                'can_call' => $canCallAgent
            ] : null,
            'invoice_url' => $this->getInvoiceUrl($order),
            'can_rate' => $order->status === 'completed',
            'ratings' => [
                'agent' => $order->agentRating,
                'products' => $order->productReviews
            ]
        ]);
    }

    /**
     * Helper to get Invoice URL (generates route that handles creation)
     */
    protected function getInvoiceUrl($order)
    {
        // We point to the API route which will handle generation/serving
        return route('api.customer.orders.invoice', $order->id);
    }

    /**
     * Generate and Download Order Invoice PDF
     * Saves to storage/app/public/invoices and returns file.
     */
    public function downloadInvoice(Request $request, $id)
    {
        $order = $request->user()->orders()->findOrFail($id);

        $filename = 'invoice-' . $order->order_number . '.pdf';
        $path = 'invoices/' . $filename;

        // Check if exists
        if (!\Illuminate\Support\Facades\Storage::disk('public')->exists($path)) {
            // Generate it
            $order->load(['items.product', 'items.variation', 'shippingAddress', 'billingAddress']);
            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('pdf.invoice', compact('order'));

            // Save to public storage
            \Illuminate\Support\Facades\Storage::disk('public')->put($path, $pdf->output());
        }

        // Return file download response from storage
        $fullPath = \Illuminate\Support\Facades\Storage::disk('public')->path($path);

        return response()->download($fullPath);
    }

    /**
     * Submit ratings for an order.
     */
    public function rate(Request $request, $id)
    {
        $order = $request->user()->orders()->findOrFail($id);

        if ($order->status !== 'completed') {
            return response()->json(['message' => 'You can only rate delivered orders.'], 403);
        }

        $request->validate([
            'agent_rating' => 'nullable|integer|between:1,5',
            'agent_comment' => 'nullable|string|max:1000',
            'product_ratings' => 'required|array',
            'product_ratings.*.id' => 'required|exists:products,id',
            'product_ratings.*.rating' => 'required|integer|between:1,5',
            'product_ratings.*.comment' => 'nullable|string|max:1000',
        ]);

        // Rate Agent
        if ($request->filled('agent_rating') && $order->delivery_agent_id) {
            AgentRating::updateOrCreate(
                ['order_id' => $order->id, 'delivery_agent_id' => $order->delivery_agent_id],
                [
                    'user_id' => Auth::id(),
                    'rating' => $request->agent_rating,
                    'comment' => $request->agent_comment
                ]
            );
        }

        // Rate Products
        foreach ($request->product_ratings as $prodRate) {
            ProductReview::updateOrCreate(
                ['order_id' => $order->id, 'product_id' => $prodRate['id']],
                [
                    'user_id' => Auth::id(),
                    'rating' => $prodRate['rating'],
                    'comment' => $prodRate['comment']
                ]
            );
        }

        return response()->json(['message' => 'Review submitted successfully.']);
    }
    public function cancel(Request $request, $id)
    {
        $order = $request->user()->orders()->findOrFail($id);

        if (in_array($order->status, ['shipped', 'delivered', 'completed', 'cancelled', 'returned'])) {
            return response()->json(['message' => 'Order cannot be cancelled at this stage.'], 403);
        }

        $request->validate([
            'cancellation_reason' => 'nullable|string|max:500'
        ]);

        $order->status = 'cancelled';
        $order->cancellation_reason = $request->cancellation_reason;
        $order->save();

        return response()->json(['message' => 'Order cancelled successfully.']);
    }

    public function reorder(Request $request, $id)
    {
        $order = $request->user()->orders()->with('items.product', 'items.variation')->findOrFail($id);

        $cart = Cart::firstOrCreate(['user_id' => $request->user()->id]);
        $messages = [];
        $addedCount = 0;

        foreach ($order->items as $item) {
            $product = $item->product;

            // Check if product exists and is active
            if (!$product || $product->status == 0) {
                $messages[] = "Product '" . ($product ? $product->name : 'Unknown') . "' is no longer available.";
                continue;
            }

            $variationId = null;
            if (isset($item->options['variation_id'])) {
                $variationId = $item->options['variation_id'];
            }

            $quantity = $item->quantity;
            $price = $product->sp;
            $variation = null;

            if ($variationId) {
                $variation = ProductVariation::find($variationId);

                if (!$variation) {
                    $messages[] = "Variation for '" . $product->name . "' is no longer available.";
                    continue;
                }
                $price = $variation->sp;

                // Check stock
                if ($variation->stock < $quantity) {
                    if ($variation->stock > 0) {
                        $quantity = $variation->stock;
                        $attrVal = $item->options['attributes'][0]['value'] ?? 'Variation';
                        $messages[] = "Added only {$quantity} of '{$product->name} - {$attrVal}' due to stock limits.";
                    } else {
                        $attrVal = $item->options['attributes'][0]['value'] ?? 'Variation';
                        $messages[] = "'{$product->name} - {$attrVal}' is out of stock.";
                        continue;
                    }
                }
            }

            // Add to cart
            $existing = $cart->items()
                ->where('product_id', $product->id)
                ->where('product_variation_id', $variationId)
                ->first();

            if ($existing) {
                $existing->quantity += $quantity;
                // Cap at stock if variation exists
                if ($variation && $existing->quantity > $variation->stock) {
                    $existing->quantity = $variation->stock;
                }
                $existing->save();
            } else {
                $cart->items()->create([
                    'product_id' => $product->id,
                    'product_variation_id' => $variationId,
                    'quantity' => $quantity,
                    'price' => $price
                ]);
            }
            $addedCount++;
        }

        if ($addedCount > 0) {
            $response = [
                'message' => 'Items added to cart.',
                'status' => 'success',
                'cart_item_count' => $cart->items()->count()
            ];

            if (count($messages) > 0) {
                $response['message'] = "Items added with some availability adjustments.";
                $response['adjustments'] = $messages;
            }

            return response()->json($response);
        } else {
            return response()->json([
                'message' => 'Could not reorder items.',
                'errors' => $messages
            ], 400);
        }
    }
}
