<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Notifications\DatabaseNotification;

class CustomerNotificationController extends Controller
{
    /**
     * List all notifications for the customer.
     */
    public function index(Request $request)
    {
        $user = $request->user();

        $notifications = $user->notifications()->latest()->paginate(20);

        // Transform collection to a friendly format
        $notifications->getCollection()->transform(function ($notification) {
            return [
                'id' => $notification->id,
                'type' => class_basename($notification->type),
                'title' => $notification->data['title'] ?? 'Notification',
                'body' => $notification->data['message'] ?? $notification->data['body'] ?? '',
                'data' => $notification->data, // Keep full data available if needed
                'read_at' => $notification->read_at,
                'is_read' => !is_null($notification->read_at),
                'created_at' => $notification->created_at,
                'created_at_human' => $notification->created_at->diffForHumans(),
            ];
        });

        return response()->json([
            'meta' => [
                'unread_count' => $user->unreadNotifications()->count(),
                'total_count' => $user->notifications()->count(),
                'current_page' => $notifications->currentPage(),
                'last_page' => $notifications->lastPage(),
                'per_page' => $notifications->perPage(),
            ],
            'data' => $notifications->items(),
        ]);
    }

    /**
     * Mark a specific notification as read.
     */
    public function markAsRead(Request $request, $id)
    {
        $notification = $request->user()->notifications()->find($id);

        if (!$notification) {
            return response()->json(['message' => 'Notification not found.'], 404);
        }

        if (!$notification->read_at) {
            $notification->markAsRead();
        }

        return response()->json([
            'message' => 'Notification marked as read.',
            'id' => $notification->id,
            'is_read' => true,
            'read_at' => now()
        ]);
    }

    /**
     * Mark all notifications as read.
     */
    public function markAllAsRead(Request $request)
    {
        $request->user()->unreadNotifications->markAsRead();

        return response()->json([
            'message' => 'All notifications marked as read.',
            'timestamp' => now()
        ]);
    }

    /**
     * Delete a specific notification.
     */
    public function destroy(Request $request, $id)
    {
        $notification = $request->user()->notifications()->find($id);

        if (!$notification) {
            return response()->json(['message' => 'Notification not found.'], 404);
        }

        $notification->delete();

        return response()->json([
            'message' => 'Notification deleted successfully.',
            'id' => $id
        ]);
    }

    /**
     * Clear all notifications.
     */
    public function clearAll(Request $request)
    {
        $count = $request->user()->notifications()->delete();

        return response()->json([
            'message' => 'All notifications cleared.',
            'deleted_count' => $count
        ]);
    }
}
