<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\GiftCard;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class CustomerGiftCardController extends Controller
{
    /**
     * List my generated gift cards.
     */
    public function index(Request $request)
    {
        $user = $request->user();

        $giftCards = GiftCard::where(function ($query) use ($user) {
            $query->where('created_by', $user->id)
                ->orWhere('recipient_email', $user->email);
        })
            ->latest()
            ->paginate(10);

        return response()->json($giftCards);
    }

    /**
     * Create (Purchase) a new gift card.
     * This deducts from wallet immediately.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:1',
            'recipient_email' => 'nullable|email',
        ]);

        $user = $request->user();
        $amount = $validated['amount'];

        if ($user->wallet_balance < $amount) {
            return response()->json(['message' => 'Insufficient wallet balance.'], 400);
        }

        DB::beginTransaction();

        try {
            // Deduct from wallet
            $user->debitWallet($amount, "Purchased Gift Card", 'App\Models\GiftCard'); // Reference updated properly later or handled here

            // Create Gift Card
            $giftCard = GiftCard::create([
                'code' => strtoupper(Str::random(10)),
                'amount' => $amount,
                'status' => 'active',
                'created_by' => $user->id,
                'recipient_email' => $request->recipient_email,
                'expires_at' => now()->addYear(), // Default 1 year expiry
            ]);

            // Update transaction reference
            // Since debitWallet creates a transaction, fetching the latest one to update reference ID if needed is tricky without return.
            // But usually User model's debitWallet should return the transaction or we can just pass an ID if available. 
            // In the User model we saw `debitWallet` doesn't return anything. 
            // For now, reliance on description is okay, or we can manually update the last transaction.
            $txn = $user->walletTransactions()->latest()->first();
            if ($txn && $txn->amount == $amount && $txn->type == 'debit') {
                $txn->update(['reference_id' => $giftCard->id, 'reference_type' => 'App\Models\GiftCard']);
            }

            DB::commit();

            return response()->json([
                'message' => 'Gift card created successfully.',
                'gift_card' => $giftCard
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Failed to create gift card: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Redeem a gift card code into wallet.
     */
    public function redeem(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string|exists:gift_cards,code',
        ]);

        $giftCard = GiftCard::where('code', $validated['code'])->first();

        if ($giftCard->status !== 'active') {
            return response()->json(['message' => 'This gift card is already ' . $giftCard->status . '.'], 400);
        }

        if ($giftCard->expires_at && $giftCard->expires_at < now()) {
            $giftCard->update(['status' => 'expired']);
            return response()->json(['message' => 'This gift card has expired.'], 400);
        }

        // Prevent creator from redeeming their own card (Optional rule, mostly to prevent churning)
        if ($giftCard->created_by == $request->user()->id) {
            // You might allow this or block it based on business logic. 
            // Usually redeeming your own gift card is pointless unless it's a workaround for something.
            // Let's allow it for now unless specified otherwise.
        }

        DB::beginTransaction();
        try {
            // Update Gift Card
            $giftCard->update([
                'status' => 'redeemed',
                'redeemed_by' => $request->user()->id,
                'redeemed_at' => now(),
            ]);

            // Credit User Wallet
            $request->user()->creditWallet(
                $giftCard->amount,
                "Redeemed Gift Card: " . $giftCard->code,
                'App\Models\GiftCard',
                $giftCard->id
            );

            DB::commit();

            return response()->json([
                'message' => 'Gift card redeemed successfully.',
                'amount' => $giftCard->amount
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Redemption failed: ' . $e->getMessage()], 500);
        }
    }
}
