<?php

namespace App\Http\Controllers\Api;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Cart;
use App\Models\Address;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\GeneralSetting;
use App\Models\Product;
use App\Models\ProductVariation;
use App\Models\Coupon;
use App\Models\GiftCard;
use App\Services\OrderMailService; // Assuming this service exists based on CheckoutController
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Razorpay\Api\Api as RazorpayApi;

class CustomerCheckoutController extends Controller
{
    /**
     * Preview checkout (Calculate totals, shipping, discounts).
     */
    public function preview(Request $request)
    {
        $request->validate([
            'address_id' => 'required|exists:addresses,id',
            'coupon_code' => 'nullable|string',
            'gift_card_code' => 'nullable|string',
            'redeem_points' => 'boolean'
        ]);

        $user = $request->user();
        $cart = Cart::where('user_id', $user->id)->with(['items.product', 'items.variation'])->first();

        if (!$cart || $cart->items->isEmpty()) {
            return response()->json(['message' => 'Cart is empty.'], 400);
        }

        $address = Address::where('user_id', $user->id)->find($request->address_id);
        if (!$address) {
            return response()->json(['message' => 'Invalid address selected.'], 400);
        }

        $calculation = $this->calculateOrderTotals($cart, $address, $user, $request);

        return response()->json([
            'summary' => $calculation
        ]);
    }

    /**
     * Place the order.
     */
    public function placeOrder(Request $request)
    {
        $request->validate([
            'address_id' => 'required|exists:addresses,id',
            'payment_method' => 'required|in:cod,razorpay,phonepe',
            'coupon_code' => 'nullable|string',
            'gift_card_code' => 'nullable|string',
            'redeem_points' => 'boolean'
        ]);

        $user = $request->user();
        $cart = Cart::where('user_id', $user->id)->with(['items.product', 'items.variation'])->first();

        if (!$cart || $cart->items->isEmpty()) {
            return response()->json(['message' => 'Cart is empty.'], 400);
        }

        $address = Address::where('user_id', $user->id)->find($request->address_id);
        if (!$address) {
            return response()->json(['message' => 'Invalid address selected.'], 400);
        }

        // Calculate final totals
        $calc = $this->calculateOrderTotals($cart, $address, $user, $request);

        DB::beginTransaction();
        try {
            // Create Order
            $order = Order::create([
                'user_id' => $user->id,
                'order_number' => 'ORD-' . strtoupper(Str::random(10)),
                'status' => 'pending_payment',
                'subtotal' => $calc['subtotal'],
                'shipping_cost' => $calc['shipping'],
                'discount' => $calc['discount'],
                'gift_card_id' => $calc['gift_card_id'],
                'gift_card_discount' => $calc['gift_card_discount'],
                'points_redeemed' => $calc['points_redeemed'],
                'points_discount' => $calc['points_discount'],
                'total' => $calc['total'],
                'total_amount' => $calc['total'], // Alias
                'payment_status' => 'pending',
                'shipping_address_id' => $address->id,
                'billing_address_id' => $address->id,
                'email' => $user->email,
                'phone' => $address->phone, // Use address phone for delivery contact
                'name' => $address->name,
                'payment_method' => $request->payment_method,
                'shipping_tax_amount' => $calc['shipping_tax_amount'],
                'shipping_tax_rate' => $calc['shipping_tax_rate']
            ]);

            // Create Items
            foreach ($cart->items as $item) {
                $price = $item->variation ? ($item->variation->sp ?? $item->variation->mrp) : ($item->product->sp ?? $item->product->mrp);
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $item->product_id,
                    'quantity' => $item->quantity,
                    'price' => $price,
                    'total' => $item->quantity * $price,
                    'options' => $item->variation ? [
                        'variation_id' => $item->variation->id,
                        'attributes' => $item->variation->options->map(function ($opt) {
                            return ['attribute' => $opt->attribute_name, 'value' => $opt->attribute_value];
                        })
                    ] : null,
                ]);
            }

            // Debit Points for redemption immediately
            if ($calc['points_redeemed'] > 0) {
                $user->debitWallet($calc['points_redeemed'], "Redeemed for Order #" . $order->order_number, 'App\Models\Order', $order->id);
            }

            DB::commit();

            // Handle Payment Methods
            if ($request->payment_method === 'cod') {
                if (!$calc['is_cod_eligible']) {
                    throw new \Exception($calc['cod_message'] ?? 'Cash on Delivery is unavailable for this order.');
                }
                return $this->processCOD($order);
            } elseif ($request->payment_method === 'razorpay') {
                return $this->initiateRazorpay($order);
            } elseif ($request->payment_method === 'phonepe') {
                // Return Order ID for frontend to initiate via SDK or WebView
                // Usually Server generates checksum and returns params.
                // We'll reimplement initiatePhonePe logic to return JSON params.
                return $this->initiatePhonePe($order);
            }

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Failed to place order: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Verify Payment (Razorpay/PhonePe)
     */
    public function verifyPayment(Request $request)
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id',
            'payment_method' => 'required|in:razorpay,phonepe',
            // Razorpay params
            'razorpay_payment_id' => 'required_if:payment_method,razorpay',
            'razorpay_order_id' => 'required_if:payment_method,razorpay',
            'razorpay_signature' => 'required_if:payment_method,razorpay',
            // PhonePe params (usually handled via webhook or status check, but app might send success signal)
        ]);

        $order = Order::where('user_id', $request->user()->id)->find($request->order_id);
        if (!$order) {
            return response()->json(['message' => 'Order not found.'], 404);
        }

        if ($request->payment_method === 'razorpay') {
            return $this->verifyRazorpay($request, $order);
        }

        // PhonePe verification usually needs server-to-server call using transaction ID
        // We can add a method for that if frontend passes transaction ID

        return response()->json(['message' => 'Invalid payment method for verification.'], 400);
    }


    // --- Private Processors ---

    private function calculateOrderTotals(Cart $cart, Address $address, $user, Request $request)
    {
        $settings = GeneralSetting::firstOrNew();

        // 1. Subtotal
        $subtotal = $cart->items->sum(function ($item) {
            $price = $item->variation ? ($item->variation->sp ?? $item->variation->mrp) : ($item->product->sp ?? $item->product->mrp);
            return $item->quantity * $price;
        });

        // 2. Shipping
        $shipping = 0;
        $isFreeShipping = false;

        if ($settings->free_shipping_enabled && $settings->min_order_amount_for_free_shipping > 0) {
            if ($subtotal >= $settings->min_order_amount_for_free_shipping) {
                $isFreeShipping = true;
            }
        }

        if (!$isFreeShipping) {
            $shipping = 0;

            // Get current store
            $currentStore = null;
            if ($settings->current_store_id) {
                $currentStore = \App\Models\Store::find($settings->current_store_id);
            }

            if ($currentStore) {
                $userCity = trim(strtolower($address->city));
                $userCountry = trim(strtolower($address->country));
                $storeCity = trim(strtolower($currentStore->city));
                $storeCountry = trim(strtolower($currentStore->country));

                if ($userCountry !== $storeCountry) {
                    $shipping = $settings->shipping_international_amount ?? 0;
                } elseif ($userCity !== $storeCity) {
                    $shipping = $settings->shipping_national_amount ?? 0;
                } else {
                    $shipping = $settings->shipping_local_amount ?? 0;
                }
            } else {
                $country = strtolower($address->country ?? 'india');
                $shipping = ($country === 'india') ? ($settings->shipping_national_amount ?? 0) : ($settings->shipping_international_amount ?? 1000);
            }
        }


        $discount = 0;
        $couponCode = null;
        $couponError = null;

        // 3. Coupon
        if ($request->coupon_code) {
            $code = strtoupper(trim($request->coupon_code));
            $coupon = Coupon::where('code', $code)->first();

            if (!$coupon) {
                $couponError = 'Invalid coupon code.';
            } elseif (!$coupon->isValidFor($subtotal)) {
                if ($coupon->min_spend > 0 && $subtotal < $coupon->min_spend) {
                    $couponError = "Minimum spend of ₹" . number_format($coupon->min_spend, 2) . " required.";
                } else {
                    $couponError = "Coupon is expired or not applicable.";
                }
            } else {
                $discount = $coupon->calculateDiscount($subtotal);

                // Fallback manual calculation if model method returns null (safety)
                if (!$discount) {
                    if ($coupon->type === 'percentage') {
                        $discount = ($subtotal * $coupon->value) / 100;
                    } else {
                        $discount = $coupon->value;
                    }
                }

                if ($discount > $subtotal)
                    $discount = $subtotal;
                $couponCode = $coupon->code;
            }
        }

        // 4. Gift Card
        $giftCardDiscount = 0;
        $giftCardId = null;
        if ($request->gift_card_code) {
            if ($couponCode)
                throw new \Exception("Cannot combine Coupon and Gift Card.");

            $gc = GiftCard::where('code', strtoupper(trim($request->gift_card_code)))->active()->first();
            if ($gc) {
                $giftCardId = $gc->id;
                $payable = $subtotal + $shipping - $discount;
                $giftCardDiscount = min($payable, $gc->amount);
            }
        }

        // 5. Points
        $pointsRedeemed = 0;
        $pointsDiscount = 0;
        if ($request->boolean('redeem_points') && $settings->wallet_enabled) {
            $balance = $user->wallet_balance;
            if ($balance >= $settings->wallet_min_points_redeem) {
                $payable = $subtotal + $shipping - $discount - $giftCardDiscount;
                if ($payable > 0) {
                    $rate = $settings->wallet_exchange_rate > 0 ? $settings->wallet_exchange_rate : 1;
                    // Logic: 1 Unit Currency = X Points? Or 1 Point = X Currency?
                    // Creating Order logic says: redeem_points = ceil(currentTotal * rate).
                    // This implies Rate = Points per 1 Currency.

                    $maxDiscountPossible = $balance / $rate;

                    if ($maxDiscountPossible >= $payable) {
                        $pointsDiscount = $payable;
                        $pointsRedeemed = ceil($payable * $rate);
                    } else {
                        $pointsDiscount = number_format($maxDiscountPossible, 2, '.', '');
                        $pointsRedeemed = $balance;
                    }
                }
            }
        }

        $total = $subtotal + $shipping - $discount - $giftCardDiscount - $pointsDiscount;
        if ($total < 0)
            $total = 0;

        $walletConfig = [
            'enabled' => (bool) $settings->wallet_enabled,
            'balance' => $user->wallet_balance ?? 0,
            'exchange_rate' => $settings->wallet_exchange_rate ?? 1,
            'min_points_required' => $settings->wallet_min_points_redeem ?? 0,
            'balance_value' => ($user->wallet_balance ?? 0) / ($settings->wallet_exchange_rate > 0 ? $settings->wallet_exchange_rate : 1),
            'min_required_error' => ($user->wallet_balance < $settings->wallet_min_points_redeem) ? "Minimum {$settings->wallet_min_points_redeem} points required to redeem." : null
        ];

        // Fetch available coupons
        $availableCoupons = Coupon::where('is_active', true)
            ->where(function ($query) {
                $query->whereNull('valid_from')
                    ->orWhere('valid_from', '<=', now());
            })
            ->where(function ($query) {
                $query->whereNull('valid_until')
                    ->orWhere('valid_until', '>=', now());
            })
            ->get()
            ->filter(function ($coupon) {
                // Manual check for usage limits since we relaxed isValidFor
                if ($coupon->usage_limit > 0 && $coupon->used_count >= $coupon->usage_limit) {
                    return false;
                }
                return true;
            })
            ->map(function ($coupon) use ($subtotal) {
                $isApplicable = true;
                $message = null;
                $missingAmount = 0;

                if ($coupon->min_spend > 0 && $subtotal < $coupon->min_spend) {
                    $isApplicable = false;
                    $missingAmount = $coupon->min_spend - $subtotal;
                    $message = "Add items worth ₹" . number_format($missingAmount, 2) . " to apply";
                }

                return [
                    'code' => $coupon->code,
                    'description' => $coupon->description ?? "Get " . ($coupon->type == 'percentage' ? $coupon->value . "%" : "₹" . $coupon->value) . " off",
                    'type' => $coupon->type,
                    'value' => $coupon->value,
                    'min_spend' => $coupon->min_spend,
                    'is_applicable' => $isApplicable,
                    'message' => $message,
                    'missing_amount' => $missingAmount
                ];
            })->values();

        // Calculate COD Eligibility
        $isCodEligible = true;
        $codMessage = null;

        if (!$settings->cod_enabled) {
            $isCodEligible = false;
            $codMessage = "Cash on Delivery is currently disabled.";
        }

        if ($address && strtolower($address->country) !== 'india') {
            $isCodEligible = false;
            $codMessage = "Cash on Delivery is not available for international orders.";
        }

        if ($settings->min_cod_amount > 0 && $total < $settings->min_cod_amount) {
            $isCodEligible = false;
            $codMessage = "Order total is below the minimum amount (₹" . number_format($settings->min_cod_amount, 2) . ") required for COD.";
        }

        // Calculate Shipping Tax
        $shippingTaxAmount = 0;
        $shippingTaxRate = 0;

        if ($settings->shipping_tax_enabled && $shipping > 0 && $settings->shipping_tax_rate_id) {
            $rateObj = \App\Models\TaxRate::find($settings->shipping_tax_rate_id);
            if ($rateObj) {
                $shippingTaxRate = $rateObj->rate;
                // Inclusive Tax: Tax = Shipping * (Rate / (100 + Rate))
                $shippingTaxAmount = $shipping * ($shippingTaxRate / (100 + $shippingTaxRate));
            }
        }

        return [
            'is_cod_eligible' => $isCodEligible,
            'cod_message' => $codMessage,
            'subtotal' => $subtotal,
            'shipping' => $shipping,
            'shipping_tax_amount' => $shippingTaxAmount,
            'shipping_tax_rate' => $shippingTaxRate,
            'discount' => $discount,
            'coupon_code' => $couponCode,
            'coupon_error' => $couponError,
            'gift_card_discount' => $giftCardDiscount,
            'gift_card_id' => $giftCardId,
            'points_redeemed' => $pointsRedeemed,
            'points_discount' => $pointsDiscount,
            'total' => $total,
            'wallet' => $walletConfig, // Return wallet details to frontend
            'available_coupons' => $availableCoupons,
            'available_gift_cards' => \App\Models\GiftCard::where('status', 'active')
                ->where(function ($q) use ($user) {
                    $q->where('created_by', $user->id)
                        ->orWhere('recipient_email', $user->email);
                })
                ->where(function ($q) {
                    $q->whereNull('expires_at')->orWhere('expires_at', '>', now());
                })
                ->get()
                ->map(function ($gc) {
                    return [
                        'code' => $gc->code,
                        'amount' => $gc->amount,
                        'expiry' => $gc->expires_at ? $gc->expires_at->format('Y-m-d') : null
                    ];
                })
        ];
    }

    private function processCOD(Order $order)
    {
        DB::transaction(function () use ($order) {
            $order->update([
                'status' => 'pending',
                'payment_status' => 'pending'
            ]);

            // Deduct Stock
            $this->deductStock($order);
            // Redeem Gift Card
            $this->finalizeGiftCard($order);
            // Clear Cart
            Cart::where('user_id', $order->user_id)->delete();

            // Email (Optional, can be queued)
            // OrderMailService::send(...) 
        });

        return response()->json([
            'message' => 'Order placed successfully.',
            'order_id' => $order->id,
            'status' => 'success'
        ]);
    }

    private function initiateRazorpay(Order $order)
    {
        $settings = GeneralSetting::firstOrNew();
        $api = new RazorpayApi(trim($settings->razorpay_key_id), trim($settings->razorpay_key_secret));

        $razorpayOrder = $api->order->create([
            'receipt' => $order->order_number,
            'amount' => (int) ($order->total * 100),
            'currency' => 'INR',
            'payment_capture' => 1
        ]);

        $order->update(['transaction_id' => $razorpayOrder['id']]); // Store temporarily

        return response()->json([
            'message' => 'Payment initiated.',
            'order_id' => $order->id,
            'razorpay_order_id' => $razorpayOrder['id'],
            'amount' => $order->total * 100,
            'key_id' => $settings->razorpay_key_id
        ]);
    }

    private function verifyRazorpay(Request $request, Order $order)
    {
        $settings = GeneralSetting::firstOrNew();
        $api = new RazorpayApi(trim($settings->razorpay_key_id), trim($settings->razorpay_key_secret));

        try {
            $api->utility->verifyPaymentSignature([
                'razorpay_order_id' => $request->razorpay_order_id,
                'razorpay_payment_id' => $request->razorpay_payment_id,
                'razorpay_signature' => $request->razorpay_signature
            ]);

            DB::transaction(function () use ($order, $request, $settings) {
                $order->update([
                    'status' => 'pending',
                    'payment_status' => 'paid',
                    'transaction_id' => $request->razorpay_payment_id,
                    'payment_method' => 'razorpay'
                ]);

                $this->deductStock($order);
                $this->finalizeGiftCard($order);
                $this->awardPoints($order, $settings);
                Cart::where('user_id', $order->user_id)->delete();
            });

            return response()->json(['message' => 'Payment successful. Order placed.']);

        } catch (\Exception $e) {
            return response()->json(['message' => 'Signature verification failed: ' . $e->getMessage()], 400);
        }
    }

    // --- Helpers ---
    private function deductStock($order)
    {
        foreach ($order->items as $item) {
            if ($item->options && isset($item->options['variation_id'])) {
                ProductVariation::where('id', $item->options['variation_id'])->decrement('stock', $item->quantity);
            } else {
                // Try updating base product stock or default variation if applicable
                // Assuming logic from CheckoutController:
                $defVar = ProductVariation::where('product_id', $item->product_id)->first();
                if ($defVar)
                    $defVar->decrement('stock', $item->quantity);
            }
        }
    }

    private function finalizeGiftCard($order)
    {
        if ($order->gift_card_id) {
            $gc = GiftCard::find($order->gift_card_id);
            if ($gc && $gc->status === 'active') {
                $gc->update([
                    'status' => 'redeemed',
                    'redeemed_at' => now(),
                    'redeemed_by' => $order->user_id
                ]);
                // Credit remainder to user wallet if any
                $remainder = $gc->amount - $order->gift_card_discount;
                if ($remainder > 0) {
                    // Credit wallet implementation...
                    $user = $order->user;
                    // Assuming simple 1:1 or settings conversion
                    if ($user)
                        $user->creditWallet($remainder, "Gift Card Remainder", "App\Models\GiftCard", $gc->id);
                }
            }
        }
    }

    private function awardPoints($order, $settings)
    {
        if ($settings->wallet_enabled && $order->user_id && $settings->wallet_earn_rate > 0) {
            $points = floor(($order->total / 100) * $settings->wallet_earn_rate);
            if ($points > 0) {
                $order->user->creditWallet($points, "Earned from Order #" . $order->order_number, "App\Models\Order", $order->id);
            }
        }
    }

    private function initiatePhonePe($order)
    {
        // Return basic message for now as PhonePe S2S structure is complex.
        // Frontend can use the calculated amount and Order ID to initiate SDK.
        return response()->json([
            'message' => 'Initiate PhonePe SDK',
            'order_id' => $order->id,
            'amount' => $order->total,
            'txn_id' => 'TXN-' . $order->order_number // Example
        ]);
    }
}
