<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Cart;
use App\Models\CartItem;
use App\Models\Product;
use App\Models\ProductVariation;
use Illuminate\Support\Facades\DB;

class CustomerCartController extends Controller
{
    /**
     * Get the current user's cart.
     */
    public function index(Request $request)
    {
        $cart = $this->getCart($request->user());

        $cart->load(['items.product.images', 'items.variation']);

        $total = $cart->items->sum(function ($item) {
            return $item->price * $item->quantity;
        });

        return response()->json([
            'cart' => $cart,
            'total_amount' => $total,
            'formatted_total' => '₹' . number_format($total, 2)
        ]);
    }

    /**
     * Add an item to the cart.
     */
    public function add(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'product_variation_id' => 'nullable|exists:product_variations,id',
            'quantity' => 'required|integer|min:1'
        ]);

        $user = $request->user();
        $cart = $this->getCart($user);

        $product = Product::findOrFail($request->product_id);
        $price = $product->sp ?? $product->mrp;

        if ($request->product_variation_id) {
            $variation = ProductVariation::findOrFail($request->product_variation_id);
            $price = $variation->sp ?? $variation->mrp;
        }

        $cartItem = $cart->items()
            ->where('product_id', $request->product_id)
            ->where('product_variation_id', $request->product_variation_id)
            ->first();

        if ($cartItem) {
            $cartItem->quantity += $request->quantity;
            $cartItem->price = $price; // Update to current price
            $cartItem->save();
        } else {
            $cartItem = $cart->items()->create([
                'product_id' => $request->product_id,
                'product_variation_id' => $request->product_variation_id,
                'quantity' => $request->quantity,
                'price' => $price
            ]);
        }

        return response()->json([
            'message' => 'Item added to cart.',
            'cart_item' => $cartItem
        ], 201);
    }

    /**
     * Update item quantity.
     */
    public function update(Request $request, $itemId)
    {
        $request->validate([
            'quantity' => 'required|integer|min:1'
        ]);

        $cart = $this->getCart($request->user());
        $cartItem = $cart->items()->findOrFail($itemId);

        $cartItem->update([
            'quantity' => $request->quantity
        ]);

        return response()->json([
            'message' => 'Cart updated.',
            'cart_item' => $cartItem
        ]);
    }

    /**
     * Remove an item from the cart.
     */
    public function remove(Request $request, $itemId)
    {
        $cart = $this->getCart($request->user());
        $cartItem = $cart->items()->findOrFail($itemId);

        $cartItem->delete();

        return response()->json([
            'message' => 'Item removed from cart.'
        ]);
    }

    /**
     * Clear the entire cart.
     */
    public function clear(Request $request)
    {
        $cart = $this->getCart($request->user());
        $cart->items()->delete();

        return response()->json([
            'message' => 'Cart cleared successfully.'
        ]);
    }

    /**
     * Helper to get or create cart for user.
     */
    private function getCart($user)
    {
        return Cart::firstOrCreate(['user_id' => $user->id]);
    }
}
