<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Address;
use Illuminate\Support\Facades\Auth;

class CustomerAddressController extends Controller
{
    /**
     * List all addresses for the authenticated user.
     */
    public function index(Request $request)
    {
        $addresses = $request->user()->addresses()->latest()->get()->map(function ($address) {
            $address->pincode = $address->postal_code; // Map back for frontend
            return $address;
        });
        return response()->json($addresses);
    }

    /**
     * Store a new address.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'type' => 'required|string|in:home,office,other',
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'address_line1' => 'required|string|max:255',
            'address_line2' => 'nullable|string|max:255',
            'city' => 'required|string|max:100',
            'state' => 'required|string|max:100',
            'country' => 'nullable|string|max:100', // Added country
            'pincode' => 'required|string|max:20',
            'is_default' => 'boolean',
            'open_time' => 'nullable|date_format:H:i',
            'close_time' => 'nullable|date_format:H:i',
        ]);

        // Map pincode to postal_code for DB
        $validated['postal_code'] = $validated['pincode'];
        unset($validated['pincode']);

        if ($request->is_default) {
            // Unset other default addresses for this user
            $request->user()->addresses()->update(['is_default' => false]);
        }

        $address = $request->user()->addresses()->create($validated);
        $address->pincode = $address->postal_code; // Return expected key

        return response()->json([
            'message' => 'Address added successfully.',
            'address' => $address
        ], 201);
    }

    /**
     * Update the specified address.
     */
    public function update(Request $request, $id)
    {
        $address = $request->user()->addresses()->findOrFail($id);

        $validated = $request->validate([
            'type' => 'required|string|in:home,office,other',
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'address_line1' => 'required|string|max:255',
            'address_line2' => 'nullable|string|max:255',
            'city' => 'required|string|max:100',
            'state' => 'required|string|max:100',
            'country' => 'nullable|string|max:100', // Added country
            'pincode' => 'required|string|max:20',
            'is_default' => 'boolean',
            'open_time' => 'nullable|date_format:H:i',
            'close_time' => 'nullable|date_format:H:i',
        ]);

        // Map pincode to postal_code for DB
        $validated['postal_code'] = $validated['pincode'];
        unset($validated['pincode']);

        if ($request->is_default) {
            $request->user()->addresses()->where('id', '!=', $id)->update(['is_default' => false]);
        }

        $address->update($validated);
        $address->pincode = $address->postal_code; // Return expected key

        return response()->json([
            'message' => 'Address updated successfully.',
            'address' => $address
        ]);
    }

    /**
     * Remove the specified address.
     */
    public function destroy(Request $request, $id)
    {
        $address = $request->user()->addresses()->findOrFail($id);
        $address->delete();

        return response()->json(['message' => 'Address deleted successfully.']);
    }

    /**
     * Set an address as default.
     */
    public function setDefault(Request $request, $id)
    {
        $address = $request->user()->addresses()->findOrFail($id);

        $request->user()->addresses()->where('id', '!=', $id)->update(['is_default' => false]);
        $address->update(['is_default' => true]);

        return response()->json([
            'message' => 'Default address set successfully.',
            'address' => $address
        ]);
    }
}

