<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AdminController extends Controller
{
    public function showLogin()
    {
        if (Auth::check()) {
            return redirect()->route('admin.dashboard');
        }
        return view('admin.login');
    }

    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if (Auth::attempt($credentials)) {
            $user = Auth::user();

            if (!$user->is_active) {
                Auth::logout();
                return back()->with('error', 'Your account is inactive.');
            }

            $request->session()->regenerate();

            // Log Login History
            \App\Models\LoginHistory::create([
                'user_id' => $user->id,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'login_at' => now(),
            ]);

            return redirect()->route('admin.dashboard');
        }

        return back()->with('error', 'Invalid credentials');
    }

    public function dashboard()
    {
        if (!Auth::check()) {
            return redirect()->route('admin.login');
        }

        $counts = [
            'categories' => \App\Models\Category::where('status', true)->count(),
            'products' => \App\Models\Product::where('status', true)->count(),
            'users' => \App\Models\User::doesntHave('roles')->where('is_active', true)->count(),
            'discounts' => \App\Models\Discount::where('is_active', true)->count(),
            'pending_orders' => \App\Models\Order::where('status', 'pending')->count(),
            'processed_orders' => \App\Models\Order::where('status', 'processing')->count(),
            'shipped_orders' => \App\Models\Order::where('status', 'shipped')->count(),
            'completed_orders' => \App\Models\Order::where('status', 'completed')->count(),
            'cancelled_orders' => \App\Models\Order::where('status', 'cancelled')->count(),
            'returned_orders' => \App\Models\Order::where('status', 'returned')->count(),
            'orders' => \App\Models\Order::whereDate('created_at', today())->count(),
        ];
        $recentTickets = \App\Models\Ticket::with('user')->latest()->take(10)->get();
        $recentOrders = \App\Models\Order::with('user')->latest()->take(10)->get();
        $recentDriverRatings = \App\Models\AgentRating::with(['user', 'order', 'deliveryAgent.user'])->latest()->take(10)->get();
        $recentItemRatings = \App\Models\ProductReview::with(['user', 'product'])->latest()->take(10)->get();

        $minStockThreshold = \App\Models\GeneralSetting::value('min_stock_threshold') ?? 5; // Default to 5 if not set
        $lowStockItems = \App\Models\ProductVariation::with(['product', 'options'])
            ->where('stock', '<=', $minStockThreshold)
            ->paginate(15, ['*'], 'low_stock_page');

        return view('admin.dashboard', compact('counts', 'recentTickets', 'recentOrders', 'lowStockItems', 'recentDriverRatings', 'recentItemRatings'));
    }

    public function showChangePasswordForm()
    {
        return view('admin.profile.change-password');
    }

    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'password' => 'required|min:8|confirmed',
        ]);

        $user = Auth::user();

        if (!\Illuminate\Support\Facades\Hash::check($request->current_password, $user->password)) {
            return back()->withErrors(['current_password' => 'The provided password does not match your current password.']);
        }

        $user->update([
            'password' => \Illuminate\Support\Facades\Hash::make($request->password)
        ]);

        return back()->with('success', 'Password updated successfully.');
    }

    public function showProfileForm()
    {
        return view('admin.profile.edit');
    }

    public function updateProfile(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email,' . $user->id,
            'whatsapp_number' => 'nullable|string|max:20',
        ]);

        $user->update([
            'name' => $request->name,
            'email' => $request->email,
            'whatsapp_number' => $request->whatsapp_number,
        ]);

        return back()->with('success', 'Profile updated successfully.');
    }

    public function loginHistory()
    {
        // Fetch login history for all users where user has some specific role or just all users for now
        // Assuming we want to show all history for now
        $histories = \App\Models\LoginHistory::with('user')->latest('login_at')->paginate(20);
        return view('admin.profile.login-history', compact('histories'));
    }

    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect()->route('admin.login');
    }
}
