<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class TicketController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        // $this->middleware('role:admin'); // If implementing roles later
    }

    public function index(Request $request)
    {
        $status = $request->get('status', 'open');
        $tickets = \App\Models\Ticket::with('user', 'order')
            ->where('status', $status)
            ->latest()
            ->paginate(10);

        $counts = [
            'open' => \App\Models\Ticket::where('status', 'open')->count(),
            'closed' => \App\Models\Ticket::where('status', 'closed')->count(),
        ];

        return view('admin.tickets.index', compact('tickets', 'status', 'counts'));
    }

    public function show(\App\Models\Ticket $ticket)
    {
        // Mark notifications as read
        \Illuminate\Support\Facades\Auth::user()->unreadNotifications()
            ->where('data->ticket_id', $ticket->id)
            ->get()->markAsRead();

        $ticket->load(['messages.user', 'order', 'user']);
        return view('admin.tickets.show', compact('ticket'));
    }

    public function reply(Request $request, \App\Models\Ticket $ticket)
    {
        if ($ticket->status === 'closed') {
            return back()->with('error', 'Ticket is closed.');
        }

        $request->validate([
            'message' => 'required|string',
            'attachment' => 'nullable|file|max:2048',
        ]);

        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $attachmentPath = $request->file('attachment')->store('tickets', 'public');
        }

        $ticket->messages()->create([
            'user_id' => \Illuminate\Support\Facades\Auth::id(),
            'message' => $request->message,
            'attachment_path' => $attachmentPath,
        ]);

        // Notify User
        $ticket->user->notify(new \App\Notifications\TicketStatusUpdated($ticket, 'replied_by_admin'));

        return back()->with('success', 'Reply sent successfully.');
    }

    public function close(\App\Models\Ticket $ticket)
    {
        $ticket->update(['status' => 'closed']);

        // Notify User
        $ticket->user->notify(new \App\Notifications\TicketStatusUpdated($ticket, 'closed'));

        return back()->with('success', 'Ticket closed successfully.');
    }
}
