<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Slider;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;

class SliderController extends Controller
{
    public function index()
    {
        $sliders = Slider::orderBy('sort_order')->get();
        return view('admin.settings.sliders.index', compact('sliders'));
    }

    public function create()
    {
        return view('admin.settings.sliders.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'image' => 'required|image|max:10240', // Increased limit for high res
            'alt_text' => 'nullable|string|max:255',
            'link' => 'nullable|url|max:255',
        ], [
            'image.uploaded' => 'The image failed to upload. The file size might exceed the server limit associated with upload_max_filesize (' . ini_get('upload_max_filesize') . ').',
        ]);

        try {
            $path = $this->processImage($request->file('image'));

            Slider::create([
                'image_path' => $path,
                'alt_text' => $request->alt_text,
                'link' => $request->link,
                'status' => true,
            ]);

            return redirect()->route('admin.settings.sliders.index')->with('success', 'Slider added successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to process slider: ' . $e->getMessage())->withInput();
        }
    }

    public function edit(Slider $slider)
    {
        return view('admin.settings.sliders.edit', compact('slider'));
    }

    public function update(Request $request, Slider $slider)
    {
        $request->validate([
            'image' => 'nullable|image|max:10240',
            'alt_text' => 'nullable|string|max:255',
            'link' => 'nullable|url|max:255',
        ]);

        $data = $request->only(['alt_text', 'link']);

        if ($request->has('status')) {
            $data['status'] = $request->boolean('status');
        }

        if ($request->hasFile('image')) {
            if ($slider->image_path) {
                if (Storage::disk('public')->exists($slider->image_path)) {
                    Storage::disk('public')->delete($slider->image_path);
                }
            }
            $data['image_path'] = $this->processImage($request->file('image'));
        }

        $slider->update($data);

        return redirect()->route('admin.settings.sliders.index')->with('success', 'Slider updated successfully.');
    }

    public function toggleStatus(Slider $slider)
    {
        $slider->status = !$slider->status;
        $slider->save();
        return back()->with('success', 'Status updated.');
    }

    public function destroy(Slider $slider)
    {
        if ($slider->image_path) {
            if (Storage::disk('public')->exists($slider->image_path)) {
                Storage::disk('public')->delete($slider->image_path);
            }
        }
        $slider->delete();
        return back()->with('success', 'Slider deleted successfully.');
    }

    private function processImage($image)
    {
        $manager = new ImageManager(new Driver());

        $extension = $image->getClientOriginalExtension();
        $filename = Str::random(40) . '.' . $extension;
        $path = 'sliders/' . $filename;

        // Ensure directory exists
        if (!Storage::disk('public')->exists('sliders')) {
            Storage::disk('public')->makeDirectory('sliders');
        }

        $img = $manager->read($image);

        // Resize to 1680x550 as requested
        // Using "cover" to ensure it fills the 1680x550 box without distortion (crops excess)
        $img->cover(1680, 550);

        $encoded = $img->toJpeg(90);
        Storage::disk('public')->put($path, $encoded);

        // Return relative path for database storage (no /storage/ prefix usually stored in specific app logic, checking prev file showed plain path stored)
        // Checking prev file: $request->file('image')->store('sliders', 'public'); returns "sliders/filename.jpg"
        // So we return simple path. Accessors/View use Storage::url() or similar.

        return $path;
    }
}
