<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\GeneralSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SettingController extends Controller
{
    public function showWebsite()
    {
        $settings = GeneralSetting::firstOrNew();
        return view('admin.settings.website', compact('settings'));
    }

    public function updateWebsite(Request $request)
    {
        $request->validate([
            'site_logo' => 'nullable|image|max:2048',
            'app_icon' => 'nullable|image|max:2048',
            'favicon' => 'nullable|image|max:1024',
            'phone_sales' => 'nullable|string',
            'email_sales' => 'nullable|email',
            'instagram_feed_url' => 'nullable|string',
            'login_max_attempts' => 'nullable|integer|min:1',
            'login_decay_minutes' => 'nullable|integer|min:1',
        ]);

        $settings = GeneralSetting::firstOrNew();
        $data = $request->except([
            '_token',
            'site_logo',
            'app_icon',
            'favicon',
            'social_links',
            'social_facebook',
            'social_instagram',
            'social_twitter',
            'social_linkedin',
            'social_youtube',
            'social_pinterest'
        ]);

        // Explicitly set these if not present in request (though form sends them)
        // Actually, $request->except() covers them if they are in the request.
        // But since we want to handle defaults or ensure they are saved:
        $data['login_max_attempts'] = $request->input('login_max_attempts', 5);
        $data['login_decay_minutes'] = $request->input('login_decay_minutes', 1);

        if ($request->hasFile('site_logo')) {
            if ($settings->site_logo) {
                Storage::disk('public')->delete($settings->site_logo);
            }
            $data['site_logo'] = $request->file('site_logo')->store('settings', 'public');
        }

        if ($request->hasFile('app_icon')) {
            if ($settings->app_icon) {
                Storage::disk('public')->delete($settings->app_icon);
            }
            $data['app_icon'] = $request->file('app_icon')->store('settings', 'public');
        }

        if ($request->hasFile('favicon')) {
            if ($settings->favicon) {
                Storage::disk('public')->delete($settings->favicon);
            }
            $data['favicon'] = $request->file('favicon')->store('settings', 'public');
        }

        // Process Social Links
        $data['social_links'] = [
            'facebook' => $request->input('social_facebook'),
            'instagram' => $request->input('social_instagram'),
            'twitter' => $request->input('social_twitter'),
            'linkedin' => $request->input('social_linkedin'),
            'youtube' => $request->input('social_youtube'),
            'pinterest' => $request->input('social_pinterest'),
        ];

        $settings->fill($data)->save();

        return back()->with('success', 'Website settings updated successfully.');
    }

    public function showCart()
    {
        $settings = GeneralSetting::firstOrNew();
        return view('admin.settings.cart', compact('settings'));
    }

    public function updateCart(Request $request)
    {
        $request->validate([
            'min_order_amount_for_free_shipping' => 'nullable|numeric|min:0',
            'min_cod_amount' => 'nullable|numeric|min:0',
            'min_stock_threshold' => 'nullable|integer|min:0',
        ]);

        $settings = GeneralSetting::firstOrNew();
        $settings->cod_enabled = $request->boolean('cod_enabled');
        $settings->free_shipping_enabled = $request->has('free_shipping_enabled');
        $settings->min_order_amount_for_free_shipping = $request->input('min_order_amount_for_free_shipping');
        $settings->min_cod_amount = $request->input('min_cod_amount');
        $settings->min_stock_threshold = $request->input('min_stock_threshold');
        $settings->save();

        return back()->with('success', 'Cart settings updated successfully.');
    }

    public function showPayment()
    {
        $settings = GeneralSetting::firstOrNew();
        return view('admin.settings.payment', compact('settings'));
    }

    public function updatePayment(Request $request)
    {
        $request->validate([
            'paypal_email' => 'nullable|email',
            'paypal_link' => 'nullable|url',
            'upi_gpay_id' => 'nullable|string',
            'upi_phonepe_id' => 'nullable|string',
            'upi_bhim_id' => 'nullable|string',
            'upi_cred_id' => 'nullable|string',
            'upi_paytm_id' => 'nullable|string',
            'bank_name' => 'nullable|string',
            'bank_account_number' => 'nullable|string',
            'bank_ifsc' => 'nullable|string',
            'bank_beneficiary_name' => 'nullable|string',
            'bank_swift_code' => 'nullable|string',
        ]);

        $settings = GeneralSetting::firstOrNew();
        $fields = [
            'paypal_email',
            'paypal_link',
            'upi_gpay_id',
            'upi_phonepe_id',
            'upi_bhim_id',
            'upi_cred_id',
            'upi_paytm_id',
            'bank_name',
            'bank_account_number',
            'bank_ifsc',
            'bank_beneficiary_name',
            'bank_swift_code'
        ];

        $settings->fill($request->only($fields))->save();

        return back()->with('success', 'Payment options updated successfully.');
    }

    public function showGateway()
    {
        $settings = GeneralSetting::firstOrNew();
        return view('admin.settings.gateway', compact('settings'));
    }

    public function updateGateway(Request $request)
    {
        $request->validate([
            'razorpay_key_id' => 'nullable|string',
            'razorpay_key_secret' => 'nullable|string',
            'ccavenue_merchant_id' => 'nullable|string',
            'ccavenue_access_code' => 'nullable|string',
            'ccavenue_encryption_key' => 'nullable|string',
            'phonepe_merchant_id' => 'nullable|string',
            'phonepe_salt_key' => 'nullable|string',
            'phonepe_salt_index' => 'nullable|string',
            'phonepe_env' => 'nullable|in:UAT,PROD',
        ]);

        $settings = GeneralSetting::firstOrNew();

        $settings->razorpay_enabled = $request->has('razorpay_enabled');
        $settings->ccavenue_enabled = $request->has('ccavenue_enabled');
        $settings->phonepe_enabled = $request->has('phonepe_enabled');

        $fields = [
            'razorpay_key_id',
            'razorpay_key_secret',
            'ccavenue_merchant_id',
            'ccavenue_access_code',
            'ccavenue_encryption_key',
            'phonepe_merchant_id',
            'phonepe_salt_key',
            'phonepe_salt_index',
            'phonepe_env'
        ];

        $settings->fill($request->only($fields))->save();

        return back()->with('success', 'Payment gateway settings updated successfully.');
    }
    public function showShipping()
    {
        $settings = GeneralSetting::firstOrNew();
        return view('admin.settings.shipping', compact('settings'));
    }

    public function updateShipping(Request $request)
    {
        $request->validate([
            'shipping_local_amount' => 'nullable|numeric|min:0',
            'shipping_national_amount' => 'nullable|numeric|min:0',
            'shipping_international_amount' => 'nullable|numeric|min:0',
        ]);

        $settings = GeneralSetting::firstOrNew();
        $fields = [
            'shipping_local_amount',
            'shipping_national_amount',
            'shipping_international_amount',
        ];

        $settings->fill($request->only($fields))->save();

        return back()->with('success', 'Shipping settings updated successfully.');
    }
    public function showWallet()
    {
        $settings = GeneralSetting::firstOrNew();
        return view('admin.settings.wallet', compact('settings'));
    }

    public function updateWallet(Request $request)
    {
        $request->validate([
            'wallet_exchange_rate' => 'required|integer|min:1',
            'wallet_min_points_redeem' => 'required|integer|min:0',
            'wallet_earn_rate' => 'required|integer|min:0',
            'delivery_wallet_earn_rate' => 'nullable|integer|min:0',
            'delivery_wallet_min_redeem' => 'nullable|integer|min:0',
            'gift_card_expiry_days' => 'required|integer|min:1',
        ]);

        $settings = GeneralSetting::firstOrNew();
        $settings->wallet_enabled = $request->has('wallet_enabled');
        $settings->wallet_exchange_rate = $request->input('wallet_exchange_rate');
        $settings->wallet_min_points_redeem = $request->input('wallet_min_points_redeem');
        $settings->wallet_earn_rate = $request->input('wallet_earn_rate');

        $settings->delivery_wallet_enabled = $request->has('delivery_wallet_enabled');
        $settings->delivery_wallet_earn_rate = $request->input('delivery_wallet_earn_rate');
        $settings->delivery_wallet_min_redeem = $request->input('delivery_wallet_min_redeem');

        $settings->gift_card_enabled = $request->has('gift_card_enabled');
        $settings->gift_card_expiry_days = $request->input('gift_card_expiry_days');

        $settings->save();

        return back()->with('success', 'Wallet settings updated successfully.');
    }

    public function showStore()
    {
        $settings = GeneralSetting::firstOrNew();
        $stores = \App\Models\Store::where('is_active', true)->get();
        return view('admin.settings.store', compact('settings', 'stores'));
    }

    public function updateStore(Request $request)
    {
        $request->validate([
            'current_store_id' => 'required', // Assuming you want null to mean 'no specific store' or required to select one.
            // If you want to allow selecting 'None' (null), use 'nullable'.
        ]);

        $settings = GeneralSetting::firstOrNew();
        // Handle "None" selection if value is empty string or similar
        $storeId = $request->input('current_store_id');
        if ($storeId === '') {
            $storeId = null;
        }

        $settings->current_store_id = $storeId;
        $settings->save();

        return back()->with('success', 'Store settings updated successfully.');
    }
    public function showAppearance()
    {
        $settings = GeneralSetting::firstOrNew();
        return view('admin.settings.appearance', compact('settings'));
    }

    public function updateAppearance(Request $request)
    {
        $request->validate([
            'admin_font_size' => 'nullable|string',
            'admin_font_family' => 'nullable|string',
        ]);

        $settings = GeneralSetting::firstOrNew();
        $settings->admin_font_size = $request->input('admin_font_size');
        $settings->admin_font_family = $request->input('admin_font_family');
        $settings->save();

        return back()->with('success', 'Admin appearance settings updated successfully.');
    }
    public function showNotification()
    {
        $settings = GeneralSetting::firstOrNew();
        return view('admin.settings.notification', compact('settings'));
    }

    public function updateNotification(Request $request)
    {
        $request->validate([
            'fcm_server_key' => 'nullable|string',
            'fcm_service_account_json' => 'nullable|string',
            'fcm_project_id' => 'nullable|string',
            'mail_mailer' => 'nullable|string',
            'mail_host' => 'nullable|string',
            'mail_port' => 'nullable|string',
            'mail_username' => 'nullable|string',
            'mail_password' => 'nullable|string',
            'mail_encryption' => 'nullable|string',
            'mail_from_address' => 'nullable|email',
            'mail_from_name' => 'nullable|string',
            'mailgun_domain' => 'nullable|string',
            'mailgun_secret' => 'nullable|string',
            'mailgun_endpoint' => 'nullable|string',
        ]);

        $settings = GeneralSetting::firstOrNew();
        $settings->fcm_enabled = $request->has('fcm_enabled');
        $settings->fcm_server_key = $request->input('fcm_server_key');
        $settings->fcm_service_account_json = $request->input('fcm_service_account_json');
        $settings->fcm_project_id = $request->input('fcm_project_id');

        $settings->mail_mailer = $request->input('mail_mailer', 'smtp');
        $settings->mail_host = $request->input('mail_host');
        $settings->mail_port = $request->input('mail_port');
        $settings->mail_username = $request->input('mail_username');
        $settings->mail_password = $request->input('mail_password');
        $settings->mail_encryption = $request->input('mail_encryption');
        $settings->mail_from_address = $request->input('mail_from_address');
        $settings->mail_from_name = $request->input('mail_from_name');
        $settings->mailgun_domain = $request->input('mailgun_domain');
        $settings->mailgun_secret = $request->input('mailgun_secret');
        $settings->mailgun_endpoint = $request->input('mailgun_endpoint');

        $settings->save();

        return back()->with('success', 'Notification settings updated successfully.');
    }
    public function showLogistics()
    {
        $settings = GeneralSetting::firstOrNew();
        return view('admin.settings.logistics', compact('settings'));
    }

    public function updateLogistics(Request $request)
    {
        $request->validate([
            'shiprocket_email' => 'nullable|email',
            'shiprocket_password' => 'nullable|string',
        ]);

        $settings = GeneralSetting::firstOrNew();
        $settings->shiprocket_enabled = $request->has('shiprocket_enabled');
        $settings->shiprocket_email = $request->input('shiprocket_email');
        $settings->shiprocket_password = $request->input('shiprocket_password');

        $settings->save();

        return back()->with('success', 'Logistics settings updated successfully.');
    }
    public function showAnalytics()
    {
        $settings = GeneralSetting::firstOrNew();
        return view('admin.settings.analytics', compact('settings'));
    }

    public function updateAnalytics(Request $request)
    {
        $request->validate([
            'ga4_property_id' => 'nullable|string',
            'gsc_property_url' => 'nullable|string',
            'google_auth_json_file' => 'nullable|file|mimetypes:application/json,text/plain',
        ]);

        $settings = GeneralSetting::firstOrNew();
        $data = $request->only(['ga4_property_id', 'gsc_property_url']);

        if ($request->hasFile('google_auth_json_file')) {
            $content = file_get_contents($request->file('google_auth_json_file')->getRealPath());
            // Verify it is valid JSON
            if (json_decode($content) === null) {
                return back()->withErrors(['google_auth_json_file' => 'Invalid JSON file.']);
            }
            $data['google_auth_json'] = $content;
        }

        $settings->fill($data)->save();

        return back()->with('success', 'Analytics settings updated successfully.');
    }

    public function showTax()
    {
        $settings = GeneralSetting::firstOrNew();
        $taxRates = \App\Models\TaxRate::where('is_active', true)->get();
        return view('admin.settings.tax', compact('settings', 'taxRates'));
    }

    public function updateTax(Request $request)
    {
        $request->validate([
            'default_tax_rate_id' => 'nullable|exists:tax_rates,id',
            'shipping_tax_rate_id' => 'nullable|exists:tax_rates,id',
        ]);

        $settings = GeneralSetting::firstOrNew();
        $settings->product_tax_enabled = $request->boolean('product_tax_enabled');
        $settings->shipping_tax_enabled = $request->boolean('shipping_tax_enabled');
        $settings->default_tax_rate_id = $request->input('default_tax_rate_id');
        $settings->shipping_tax_rate_id = $request->input('shipping_tax_rate_id');

        $settings->save();

        return back()->with('success', 'Tax settings updated successfully.');
    }
}
