<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SeoSetting;
use Illuminate\Http\Request;

class SeoSettingController extends Controller
{
    public function show()
    {
        $seoSettings = SeoSetting::all()->keyBy('page_name');
        $pages = $this->getPlacementOptions();
        
        return view('admin.settings.seo', compact('seoSettings', 'pages'));
    }

    public function update(Request $request)
    {
        $data = $request->validate([
            'schemas' => 'array',
            'schemas.*' => 'nullable|string',
            'page_content' => 'array',
            'page_content.*' => 'nullable|string',
        ]);

        if (isset($data['schemas'])) {
            foreach ($data['schemas'] as $pageName => $markup) {
                SeoSetting::updateOrCreate(
                    ['page_name' => $pageName],
                    ['schema_markup' => $markup]
                );
            }
        }

        if (isset($data['page_content'])) {
            foreach ($data['page_content'] as $pageName => $content) {
                SeoSetting::updateOrCreate(
                    ['page_name' => $pageName],
                    ['page_content' => $content]
                );
            }
        }

        return back()->with('success', 'SEO settings updated successfully.');
    }

    private function getPlacementOptions()
    {
        // 1. Static Pages
        $options = collect([
            (object) ['id' => 'home', 'title' => 'Home Page', 'slug' => 'home'],
            (object) ['id' => 'shop', 'title' => 'Shop Page', 'slug' => 'shop'],
            (object) ['id' => 'about', 'title' => 'About Us Page', 'slug' => 'about'],
            (object) ['id' => 'contact', 'title' => 'Contact Us Page', 'slug' => 'contact'],
        ]);

        // 2. CMS Pages
        $cmsPages = \App\Models\Page::select('id', 'title', 'slug')->get();
        foreach ($cmsPages as $page) {
            $options->push((object) [
                'id' => 'page-' . $page->id,
                'title' => $page->title . ' (CMS Page)',
                'slug' => $page->slug
            ]);
        }

         // 3. Collection Pages (Subcategories: collections/*/*)
         $subcategories = \App\Models\Category::whereNotNull('parent_id')
         ->where('status', true)
         ->with('parent')
         ->get();

        foreach ($subcategories as $subcategory) {
            if ($subcategory->parent) {
                // Use a slug-friendly format for the key
                $slug = 'collections/' . $subcategory->parent->slug . '/' . $subcategory->slug;
                // Sanitize slug for database key if needed, but existing logic allows arbritary strings
                $options->push((object) [
                 'id' => 'cat-' . $subcategory->id,
                 'title' => $subcategory->parent->name . ' > ' . $subcategory->name . ' (Collection)',
                 'slug' => $slug
                ]);
            }
        }

        return $options;
    }
}
