<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ResellerApplication;
use Illuminate\Http\Request;

class ResellerController extends Controller
{
    public function index()
    {
        $applications = ResellerApplication::latest()->paginate(15);
        return view('admin.resellers.index', compact('applications'));
    }

    public function show(ResellerApplication $reseller)
    {
        return view('admin.resellers.show', compact('reseller'));
    }

    public function updateStatus(Request $request, ResellerApplication $reseller)
    {
        $request->validate([
            'status' => 'required|in:pending,approved,rejected',
            'admin_notes' => 'nullable|string'
        ]);

        if ($request->status === 'approved' && $reseller->status !== 'approved') {
            // 1. Create or Get User
            $password = \Illuminate\Support\Str::random(10);
            $user = \App\Models\User::firstOrCreate(
                ['email' => $reseller->email],
                [
                    'name' => $reseller->name,
                    'password' => \Illuminate\Support\Facades\Hash::make($password),
                ]
            );

            // 2. Assign Role
            $role = \App\Models\Role::firstOrCreate(['name' => 'Reseller']);
            $user->roles()->syncWithoutDetaching([$role->id]);

            // 3. Create Reseller Profile
            \App\Models\Reseller::updateOrCreate(
                ['user_id' => $user->id],
                [
                    'company_name' => $reseller->company_name,
                    'gst_number' => $reseller->gst_number,
                    'phone' => $reseller->phone,
                    'address' => $reseller->address,
                    'city' => $reseller->city,
                    'state' => $reseller->state,
                    'pincode' => $reseller->postal_code,
                    'country' => 'India',
                ]
            );

            $reseller->user_id = $user->id;
            $reseller->update([
                'status' => 'approved',
                'user_id' => $user->id,
                'admin_notes' => $request->admin_notes
            ]);

            // 4. Send Email
            try {
                \Illuminate\Support\Facades\Mail::to($user->email)->send(new \App\Mail\ResellerApproved($user, $password));
            } catch (\Exception $e) {
                // Log email error but continue
            }

            // Flash the password to the session for Admin to see once
            return back()->with('success', 'Application approved successfully. Password sent to email.')
                ->with('generated_password', $password);
        }

        $reseller->update([
            'status' => $request->status,
            'admin_notes' => $request->admin_notes,
            'user_id' => $reseller->user_id // Ensure user_id is saved if set above
        ]);

        return back()->with('success', 'Application status updated successfully.');
    }

    public function destroy(ResellerApplication $reseller)
    {
        $reseller->delete();
        return back()->with('success', 'Application deleted successfully.');
    }
}
