<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\User;
use App\Models\Store;
use Carbon\Carbon;
use Illuminate\Http\Request;

class ReportController extends Controller
{
    public function ordersReport(Request $request, $status)
    {
        $query = Order::with(['user', 'shippingAddress'])->where('status', $status);
        $this->applyFilters($request, $query);

        $orders = $query->latest()->paginate(20)->withQueryString();

        return view('admin.reports.orders', compact('orders', 'status'));
    }

    public function paymentsReport(Request $request)
    {
        $query = Order::with(['user', 'items.product']);
        $this->applyFilters($request, $query);

        $orders = $query->latest()->paginate(20)->withQueryString();

        return view('admin.reports.payments', compact('orders'));
    }

    public function downloadPaymentsPdf(Request $request)
    {
        $query = Order::with(['user', 'items.product']);
        $this->applyFilters($request, $query);
        $orders = $query->latest()->get();

        $pdf = \PDF::loadView('admin.reports.pdf.payments', compact('orders'));
        return $pdf->download('payments-report-' . now()->format('Y-m-d') . '.pdf');
    }

    public function clientsReport(Request $request)
    {
        $query = User::doesntHave('roles')->withCount('orders');

        if ($request->filled('period')) {
            $this->applyDateFilter($request->period, $query);
        } elseif ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('created_at', [$request->start_date . ' 00:00:00', $request->end_date . ' 23:59:59']);
        }

        $clients = $query->latest()->paginate(20)->withQueryString();

        return view('admin.reports.clients', compact('clients'));
    }

    public function trafficReport(Request $request)
    {
        $query = \App\Models\TrafficHistory::query();

        if ($request->filled('period')) {
            $this->applyDateFilter($request->period, $query);
        } elseif ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('created_at', [$request->start_date . ' 00:00:00', $request->end_date . ' 23:59:59']);
        }

        // Get Top Sources
        $topSources = (clone $query)
            ->select('source', \Illuminate\Support\Facades\DB::raw('count(*) as count'))
            ->groupBy('source')
            ->orderByDesc('count')
            ->get();

        // Get Top Landing Pages
        $topPages = (clone $query)
            ->select('landing_page', \Illuminate\Support\Facades\DB::raw('count(*) as count'))
            ->groupBy('landing_page')
            ->orderByDesc('count')
            ->limit(10)
            ->get();

        // Get Device Breakdown
        $deviceStats = (clone $query)
            ->select('device', \Illuminate\Support\Facades\DB::raw('count(*) as count'))
            ->groupBy('device')
            ->orderByDesc('count')
            ->get();

        $traffic = $query->latest()->paginate(20)->withQueryString();

        return view('admin.reports.traffic', compact('traffic', 'topSources', 'topPages', 'deviceStats'));
    }

    private function applyFilters(Request $request, $query)
    {
        // Scope Filter: local, national, international
        if ($request->filled('scope')) {
            $store = Store::where('is_active', true)->first();
            $storeCountry = $store ? $store->country : 'India';
            $storeState = $store ? $store->state : '';

            $query->whereHas('shippingAddress', function ($q) use ($request, $storeCountry, $storeState) {
                if ($request->scope == 'local') {
                    $q->where('country', $storeCountry);
                    if ($storeState) {
                        $q->where('state', $storeState);
                    }
                } elseif ($request->scope == 'national') {
                    $q->where('country', $storeCountry);
                    if ($storeState) {
                        $q->where('state', '!=', $storeState);
                    }
                } elseif ($request->scope == 'international') {
                    $q->where('country', '!=', $storeCountry);
                }
            });
        }

        // Date Period Filter
        if ($request->filled('period')) {
            $this->applyDateFilter($request->period, $query);
        } elseif ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('created_at', [$request->start_date . ' 00:00:00', $request->end_date . ' 23:59:59']);
        }
    }

    private function applyDateFilter($period, $query)
    {
        switch ($period) {
            case 'today':
                $query->whereDate('created_at', Carbon::today());
                break;
            case 'yesterday':
                $query->whereDate('created_at', Carbon::yesterday());
                break;
            case 'past_7_days':
                $query->where('created_at', '>=', Carbon::now()->subDays(7));
                break;
            case 'this_week':
                $query->whereBetween('created_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()]);
                break;
            case 'last_week':
                $query->whereBetween('created_at', [Carbon::now()->subWeek()->startOfWeek(), Carbon::now()->subWeek()->endOfWeek()]);
                break;
            case 'past_month':
                $query->where('created_at', '>=', Carbon::now()->subMonth());
                break;
            case 'this_month':
                $query->whereMonth('created_at', Carbon::now()->month)->whereYear('created_at', Carbon::now()->year);
                break;
            case 'last_month':
                $query->whereMonth('created_at', Carbon::now()->subMonth()->month)->whereYear('created_at', Carbon::now()->subMonth()->year);
                break;
            case 'past_year':
                $query->where('created_at', '>=', Carbon::now()->subYear());
                break;
            case 'this_year':
                $query->whereYear('created_at', Carbon::now()->year);
                break;
        }
    }

    public function shippingReport(Request $request)
    {
        $query = Order::with(['user', 'shippingAddress'])->where('status', '!=', 'cancelled');
        $this->applyFilters($request, $query);
        $orders = $query->latest()->paginate(20)->withQueryString();
        return view('admin.reports.shipping', compact('orders'));
    }

    public function downloadShippingPdf(Request $request)
    {
        $query = Order::with(['user', 'shippingAddress'])->where('status', '!=', 'cancelled');
        $this->applyFilters($request, $query);
        $orders = $query->latest()->get();

        $pdf = \PDF::loadView('admin.reports.pdf.shipping', compact('orders'));
        return $pdf->download('shipping-report-' . now()->format('Y-m-d') . '.pdf');
    }

    public function taxReport(Request $request)
    {
        $query = Order::with(['user', 'items.product'])->where('status', '!=', 'cancelled');
        $this->applyFilters($request, $query);
        $orders = $query->latest()->paginate(20)->withQueryString();

        $settings = \App\Models\GeneralSetting::first();
        $shippingTaxRate = 0;
        if ($settings && $settings->shipping_tax_enabled && $settings->shipping_tax_rate_id) {
            $rateObj = \App\Models\TaxRate::find($settings->shipping_tax_rate_id);
            if ($rateObj) {
                $shippingTaxRate = $rateObj->rate;
            }
        }

        return view('admin.reports.tax', compact('orders', 'settings', 'shippingTaxRate'));
    }

    public function downloadTaxPdf(Request $request)
    {
        $query = Order::with(['user', 'items.product'])->where('status', '!=', 'cancelled');
        $this->applyFilters($request, $query);
        $orders = $query->latest()->get();

        $settings = \App\Models\GeneralSetting::first();
        $shippingTaxRate = 0;
        if ($settings && $settings->shipping_tax_enabled && $settings->shipping_tax_rate_id) {
            $rateObj = \App\Models\TaxRate::find($settings->shipping_tax_rate_id);
            if ($rateObj) {
                $shippingTaxRate = $rateObj->rate;
            }
        }

        $pdf = \PDF::loadView('admin.reports.pdf.tax', compact('orders', 'settings', 'shippingTaxRate'));
        return $pdf->download('tax-report-' . now()->format('Y-m-d') . '.pdf');
    }
}
