<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::with(['category', 'images'])->withCount('variations')
            ->withSum([
                'orderItems as total_sold_qty' => function ($q) {
                    $q->whereHas('order', function ($oq) {
                        $oq->where('status', '!=', 'pending_payment');
                    });
                }
            ], 'quantity');

        if ($request->has('search')) {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%")
                ->orWhere('slug', 'like', "%{$search}%");
        }

        if ($request->has('category') && $request->category != '') {
            $query->where('category_id', $request->category);
        }

        // Sorting
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        if ($sort === 'category') {
            $query->join('categories', 'products.category_id', '=', 'categories.id')
                ->select('products.*') // Avoid column collisions
                ->orderBy('categories.name', $direction);
        } else {
            // Default sorting (name, price, etc. which are on products table)
            // Ensure column exists to avoid errors, though blade should control this
            if (in_array($sort, ['name', 'created_at', 'sp'])) {
                $query->orderBy('products.' . $sort, $direction);
            } else {
                $query->orderBy('products.created_at', 'desc');
            }
        }

        $products = $query->paginate(100);
        $categories = Category::whereNull('parent_id')->get(); // Fetch parent categories for filter

        return view('admin.products.index', compact('products', 'categories'));
    }

    public function create()
    {
        // Get parent categories
        $categories = Category::whereNull('parent_id')->where('status', true)->get();
        // Get Default Tax Rate Setting
        $defaultTaxRate = \App\Models\GeneralSetting::value('default_tax_rate_id');
        // Fetch actual rate value if ID exists, or pass list if needed for dropdown default selection logic in blade
        // The user wants: "WHEN WE ADD PRODUCTS, IT SHOULD CHECK DEFAULT TAX RATE"
        // This implies the relevant dropdown should pre-select this value.

        return view('admin.products.create', compact('categories', 'defaultTaxRate'));
    }

    public function getSubcategories(Category $category)
    {
        return response()->json($category->children()->where('status', true)->get());
    }

    public function duplicate(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'd_category_id' => 'required|exists:categories,id',
            'd_subcategory_id' => 'required|exists:categories,id',
        ]);

        $product = Product::with(['images', 'variations.options', 'upsells', 'crossSells'])->findOrFail($request->product_id);
        $targetSubCategory = Category::findOrFail($request->d_subcategory_id);

        $newSlug = Str::slug($product->name . ' ' . $targetSubCategory->name);
        if (Product::where('slug', $newSlug)->exists()) {
            $newSlug = $newSlug . '-' . time();
        }

        $newProduct = $product->replicate();
        $newProduct->category_id = $request->d_category_id;
        $newProduct->subcategory_id = $request->d_subcategory_id;
        $newProduct->slug = $newSlug;
        $newProduct->status = true; // Set to active by default as requested
        $newProduct->save();

        // 1. Replicate Images
        foreach ($product->images as $image) {
            $newImage = $image->replicate();
            $newImage->product_id = $newProduct->id;
            $newImage->save();
        }

        // 2. Replicate Variations
        foreach ($product->variations as $variation) {
            $newVariation = $variation->replicate();
            $newVariation->product_id = $newProduct->id;
            if ($newVariation->sku) {
                $newVariation->sku = $newVariation->sku . '-' . Str::random(4);
            }
            $newVariation->save();

            foreach ($variation->options as $option) {
                $newOption = $option->replicate();
                $newOption->product_variation_id = $newVariation->id;
                $newOption->save();
            }
        }

        // 3. Replicate Relations
        if ($product->upsells()->exists()) {
            $newProduct->upsells()->attach($product->upsells->pluck('id'), ['type' => 'upsell']);
        }
        if ($product->crossSells()->exists()) {
            $newProduct->crossSells()->attach($product->crossSells->pluck('id'), ['type' => 'cross_sell']);
        }

        return back()->with('success', 'Product duplicated successfully to the selected sub-category.');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'category_id' => 'required|exists:categories,id',
            'mrp' => 'required|numeric',
            'sp' => 'required|numeric',
            'reseller_price' => 'nullable|numeric',
            'number_of_units' => 'required|integer|min:1',
            'unit_type' => 'nullable|string',
            'short_description' => 'required',
            'dispatch_days' => 'required|integer',
            'images' => 'nullable|array',
            'images.*' => 'image|max:5120',
            'return_window_days' => 'nullable|integer|min:0',
        ]);

        $status = $request->status ? true : false;
        $slug = $request->slug ? Str::slug($request->slug) : Str::slug($request->name);
        if (Product::where('slug', $slug)->exists()) {
            $slug = $slug . '-' . time();
        }

        $product = Product::create([
            'category_id' => $request->category_id,
            'subcategory_id' => $request->subcategory_id,
            'name' => $request->name,
            'slug' => $slug,
            'mrp' => $request->mrp,
            'sp' => $request->sp,
            'reseller_price' => $request->reseller_price ?? $request->sp,
            'number_of_units' => $request->number_of_units ?? 1,
            'unit_type' => $request->unit_type,
            'tax_code' => $request->tax_code ?? (\App\Models\GeneralSetting::value('product_tax_enabled') ? '18' : 0),
            'hsn_code' => $request->hsn_code,
            'short_description' => $request->short_description,
            'long_description' => $request->long_description,
            'dispatch_days' => $request->dispatch_days,
            'country_origin' => $request->country_origin ?? 'India',
            'warranty_details' => $request->warranty_details,
            'guarantee_details' => $request->guarantee_details,
            'schema_markup' => $request->schema_markup,
            'breadcrumbs_title' => $request->breadcrumbs_title,
            'canonical_url' => $request->canonical_url,
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description,
            'meta_keywords' => $request->meta_keywords,
            'status' => $status,
            'is_returnable' => $request->has('is_returnable'),
            'return_window_days' => $request->return_window_days,
            'return_policy_desc' => $request->return_policy_desc,
        ]);

        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $index => $image) {
                $paths = $this->processImage($image, 'products');
                $product->images()->create([
                    'image_path' => $paths['original'],
                    'thumbnail_path' => $paths['thumbnail'],
                    'alt_text' => $request->name,
                    'sort_order' => $index,
                ]);
            }
        }

        // Handle Variations
        if ($request->has('variations') && is_array($request->variations) && count($request->variations) > 0) {
            foreach ($request->variations as $key => $varData) {
                $varImagePath = null;
                $varThumbnailPath = null;
                // Handle variation image upload
                if ($request->hasFile("variations.$key.image")) {
                    $varImage = $request->file("variations.$key.image");
                    $paths = $this->processImage($varImage, 'products/variations');
                    $varImagePath = $paths['original'];
                    $varThumbnailPath = $paths['thumbnail'];
                }

                $variation = $product->variations()->create([
                    'mrp' => $varData['mrp'],
                    'sp' => $varData['sp'],
                    'reseller_price' => $varData['reseller_price'] ?? $varData['sp'],
                    'stock' => $varData['stock'],
                    'sku' => $varData['sku'] ?? null,
                    'image_path' => $varImagePath,
                    'thumbnail_path' => $varThumbnailPath,
                ]);

                if (isset($varData['attributes'])) {
                    foreach ($varData['attributes'] as $attrName => $attrValue) {
                        $variation->options()->create([
                            'attribute_name' => $attrName,
                            'attribute_value' => $attrValue,
                        ]);
                    }
                }
            }
        } else {
            // Create default variation if no variations provided
            // Get first product image if available
            $firstImage = $product->images()->orderBy('sort_order')->first();
            $defaultImagePath = $firstImage ? $firstImage->image_path : null;
            $defaultThumbnailPath = $firstImage ? $firstImage->thumbnail_path : null;

            $product->variations()->create([
                'mrp' => $product->mrp,
                'sp' => $product->sp,
                'reseller_price' => $product->reseller_price,
                'stock' => 100, // Default quantity as requested
                'sku' => strtoupper(Str::slug($product->name)), // Generate SKU from name
                'image_path' => $defaultImagePath,
                'thumbnail_path' => $defaultThumbnailPath,
            ]);
        }

        // Handle Product Relations (Upsell & Cross-sell)
        if ($request->has('upsell_products')) {
            $product->upsells()->attach($request->upsell_products, ['type' => 'upsell']);
        }
        if ($request->has('cross_sell_products')) {
            $product->crossSells()->attach($request->cross_sell_products, ['type' => 'cross_sell']);
        }

        return redirect()->route('admin.products.create')->with('success', 'Product created successfully');
    }

    public function edit(Product $product)
    {
        $categories = Category::whereNull('parent_id')->where('status', true)->get();
        // Load subcategories for the selected category
        $subcategories = $product->category ? $product->category->children()->where('status', true)->get() : collect([]);

        // Eager load variations with options
        $product->load(['variations.options', 'upsells', 'crossSells']);

        // Prepare data for AlpineJS to avoid complicated Blade directives
        $variationAttributes = $product->variations->flatMap(function ($v) {
            return $v->options->pluck('attribute_name');
        })->unique()->values();

        $variationsData = $product->variations->map(function ($v) {
            return [
                'id' => $v->id,
                'mrp' => $v->mrp,
                'sp' => $v->sp,
                'reseller_price' => $v->reseller_price,
                'stock' => $v->stock,
                'sku' => $v->sku,
                'attributes' => $v->options->pluck('attribute_value', 'attribute_name'),
                'image_path' => $v->image_path ?: null
            ];
        });

        $upsellData = $product->upsells->map(fn($p) => ['id' => $p->id, 'name' => $p->name]);
        $crossSellData = $product->crossSells->map(fn($p) => ['id' => $p->id, 'name' => $p->name]);

        return view('admin.products.edit', compact('product', 'categories', 'subcategories', 'variationAttributes', 'variationsData', 'upsellData', 'crossSellData'));
    }

    public function update(Request $request, Product $product)
    {
        $request->validate([
            'name' => 'required',
            'category_id' => 'required|exists:categories,id',
            'mrp' => 'required|numeric',
            'sp' => 'required|numeric',
            'reseller_price' => 'nullable|numeric',
            'number_of_units' => 'required|integer|min:1',
            'unit_type' => 'nullable|string',
            'short_description' => 'required',
            'dispatch_days' => 'required|integer',
            'images' => 'nullable|array',
            'images.*' => 'image|max:5120',
            'return_window_days' => 'nullable|integer|min:0',
        ]);

        $status = $request->status ? true : false;

        // Slug generation logic on update
        $slug = ($request->filled('slug')) ? Str::slug($request->slug) : $product->slug;
        if ($request->filled('slug') && Product::where('slug', $slug)->where('id', '!=', $product->id)->exists()) {
            $slug = $slug . '-' . time();
        }

        $product->update([
            'category_id' => $request->category_id,
            'subcategory_id' => $request->subcategory_id,
            'name' => $request->name,
            'slug' => $slug,
            'mrp' => $request->mrp,
            'sp' => $request->sp,
            'reseller_price' => $request->reseller_price ?? $request->sp,
            'number_of_units' => $request->number_of_units ?? 1,
            'unit_type' => $request->unit_type,
            'tax_code' => $request->tax_code ?? (\App\Models\GeneralSetting::value('product_tax_enabled') ? '18' : 0),
            'hsn_code' => $request->hsn_code,
            'short_description' => $request->short_description,
            'long_description' => $request->long_description,
            'dispatch_days' => $request->dispatch_days,
            'country_origin' => $request->country_origin ?? 'India',
            'warranty_details' => $request->warranty_details,
            'guarantee_details' => $request->guarantee_details,
            'schema_markup' => $request->schema_markup,
            'breadcrumbs_title' => $request->breadcrumbs_title,
            'canonical_url' => $request->canonical_url,
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description,
            'meta_keywords' => $request->meta_keywords,
            'status' => $status,
            'is_returnable' => $request->has('is_returnable'),
            'return_window_days' => $request->return_window_days,
            'return_policy_desc' => $request->return_policy_desc,
        ]);

        if ($request->hasFile('images')) {
            $currentMaxOrder = $product->images()->max('sort_order') ?? 0;
            foreach ($request->file('images') as $index => $image) {
                $paths = $this->processImage($image, 'products');
                $product->images()->create([
                    'image_path' => $paths['original'],
                    'thumbnail_path' => $paths['thumbnail'],
                    'alt_text' => $request->name,
                    'sort_order' => $currentMaxOrder + $index + 1,
                ]);
            }
        }

        // 1. Handle Deletions
        if ($request->has('deleted_variations')) {
            \App\Models\ProductVariation::destroy($request->deleted_variations);
        }

        // 2. Handle Add/Update Variations
        if ($request->has('variations')) {
            foreach ($request->variations as $key => $varData) {
                $varImagePath = null;
                $varThumbnailPath = null;
                if ($request->hasFile("variations.$key.image")) {
                    $varImage = $request->file("variations.$key.image");
                    $paths = $this->processImage($varImage, 'products/variations');
                    $varImagePath = $paths['original'];
                    $varThumbnailPath = $paths['thumbnail'];
                }

                $variationData = [
                    'mrp' => $varData['mrp'],
                    'sp' => $varData['sp'],
                    'reseller_price' => $varData['reseller_price'] ?? $varData['sp'],
                    'stock' => $varData['stock'],
                    'sku' => $varData['sku'] ?? null,
                ];

                if ($varImagePath) {
                    $variationData['image_path'] = $varImagePath;
                    $variationData['thumbnail_path'] = $varThumbnailPath;
                }

                if (isset($varData['id']) && $varData['id']) {
                    // Update existing
                    $variation = \App\Models\ProductVariation::find($varData['id']);
                    if ($variation) {
                        $variation->update($variationData);
                    }
                } else {
                    // Create new
                    $variation = $product->variations()->create($variationData);
                }

                if ($variation && isset($varData['attributes'])) {
                    $variation->options()->delete(); // Remove old options
                    foreach ($varData['attributes'] as $attrName => $attrValue) {
                        $variation->options()->create([
                            'attribute_name' => $attrName,
                            'attribute_value' => $attrValue,
                        ]);
                    }
                }
            }
        }

        // 3. Handle Product Relations (Safe Sync)

        // Upsells
        \Illuminate\Support\Facades\DB::table('product_relations')
            ->where('product_id', $product->id)
            ->where('type', 'upsell')
            ->delete();

        if ($request->has('upsell_products') && !empty($request->upsell_products)) {
            $product->upsells()->attach($request->upsell_products, ['type' => 'upsell']);
        }

        // Cross-sells
        \Illuminate\Support\Facades\DB::table('product_relations')
            ->where('product_id', $product->id)
            ->where('type', 'cross_sell')
            ->delete();

        if ($request->has('cross_sell_products') && !empty($request->cross_sell_products)) {
            $product->crossSells()->attach($request->cross_sell_products, ['type' => 'cross_sell']);
        }

        return redirect()->route('admin.products.index')->with('success', 'Product updated successfully');
    }

    public function toggleStatus(Product $product)
    {
        $product->status = !$product->status;
        $product->save();

        return response()->json(['success' => true, 'status' => $product->status, 'message' => 'Status updated successfully']);
    }

    public function deleteImage(\App\Models\ProductImage $productImage)
    {
        // Remove '/storage/' from the beginning of the path to get the relative path
        $path = str_replace('/storage/', '', $productImage->image_path);

        if (\Illuminate\Support\Facades\Storage::disk('public')->exists($path)) {
            \Illuminate\Support\Facades\Storage::disk('public')->delete($path);
        }

        // Also delete thumbnail if exists
        if ($productImage->thumbnail_path) {
            $thumbnailPath = str_replace('/storage/', '', $productImage->thumbnail_path);
            if (\Illuminate\Support\Facades\Storage::disk('public')->exists($thumbnailPath)) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($thumbnailPath);
            }
        }

        $productImage->delete();

        return response()->json(['success' => true, 'message' => 'Image deleted successfully']);
    }

    public function searchSuggestions(Request $request)
    {
        $search = $request->query('query', '');

        if (strlen($search) < 2) {
            return response()->json([]);
        }

        $products = Product::where('name', 'like', "%{$search}%")
            ->orWhere('slug', 'like', "%{$search}%")
            ->select('id', 'name', 'slug')
            ->limit(10)
            ->get();

        return response()->json($products);
    }

    public function exportPdf()
    {
        $products = Product::with(['category', 'variations.options'])->get();
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('admin.products.pdf', compact('products'));
        // Set paper to landscape for better visibility of variations
        $pdf->setPaper('a4', 'landscape');
        return $pdf->download('products-list.pdf');
    }

    public function sales(Product $product)
    {
        $sales = $product->orderItems()
            ->with(['order.user', 'order.shippingAddress'])
            ->whereHas('order', function ($q) {
                $q->where('status', '!=', 'pending_payment');
            })
            ->latest()
            ->paginate(20);

        return view('admin.products.sales', compact('product', 'sales'));
    }

    public function downloadSample()
    {
        return \Maatwebsite\Excel\Facades\Excel::download(new \App\Exports\SampleProductsExport, 'products_sample.xlsx');
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,csv,xls',
        ]);

        try {
            \Maatwebsite\Excel\Facades\Excel::import(new \App\Imports\ProductsImport, $request->file('file'));
            return redirect()->route('admin.products.index')->with('success', 'Products imported successfully.');
        } catch (\Exception $e) {
            return redirect()->route('admin.products.index')->with('error', 'Error importing products: ' . $e->getMessage());
        }
    }

    private function processImage($image, $directory = 'products')
    {
        // Use Intervention Image v3 to generate thumbnails
        $manager = new \Intervention\Image\ImageManager(new \Intervention\Image\Drivers\Gd\Driver());

        $extension = $image->getClientOriginalExtension();
        $filename = Str::random(40) . '.' . $extension;

        // Save Original
        $originalPath = $image->storeAs($directory, $filename, 'public');

        // Prepare Thumbnail Path (WebP)
        $thumbnailDirectory = $directory . '/thumbnails';
        $filenameWithoutExt = pathinfo($filename, PATHINFO_FILENAME);
        $thumbnailFilename = $filenameWithoutExt . '.webp';
        $thumbnailPath = $thumbnailDirectory . '/' . $thumbnailFilename;

        // Ensure thumbnail directory exists
        if (!\Illuminate\Support\Facades\Storage::disk('public')->exists($thumbnailDirectory)) {
            \Illuminate\Support\Facades\Storage::disk('public')->makeDirectory($thumbnailDirectory);
        }

        try {
            // Read and process image
            // Check if we can handle this image type
            $mime = $image->getMimeType();
            if (($mime == 'image/jpeg' || $mime == 'image/jpg') && !function_exists('imagecreatefromjpeg')) {
                throw new \Exception("JPEG support missing in GD. Skipping processing.");
            }

            $img = $manager->read($image);

            // Per request: "without changing the dimensions" -> No resizing/scaling.
            // Per request: "change those images in webp" -> toWebp
            // Per request: "reduce size upto 70%" -> Quality 50 (WebP is efficient, 50% q usually yields >70% size reduction vs orig JPEG)
            $encoded = $img->toWebp(50);

            \Illuminate\Support\Facades\Storage::disk('public')->put($thumbnailPath, $encoded);

            $finalThumbnailPath = '/storage/' . $thumbnailPath;
        } catch (\Exception $e) {
            // Fallback to original image if processing fails
            $finalThumbnailPath = '/storage/' . $originalPath;
            \Illuminate\Support\Facades\Log::error('Image Processing Failed: ' . $e->getMessage());
        }

        return [
            'original' => '/storage/' . $originalPath,
            'thumbnail' => $finalThumbnailPath
        ];
    }

    public function setDefaultImage(Request $request, Product $product, $image)
    {
        $productImage = $product->images()->findOrFail($image);

        // Set this image to 0
        $productImage->update(['sort_order' => 0]);

        // Reorder others starting from 1
        $otherImages = $product->images()->where('id', '!=', $image)->orderBy('sort_order')->orderBy('id')->get();
        $count = 1;
        foreach ($otherImages as $img) {
            $img->update(['sort_order' => $count]);
            $count++;
        }

        return response()->json(['success' => true, 'message' => 'Default image set successfully.']);
    }
    public function getVariations(Product $product)
    {
        // Load variations with their options
        $product->load('variations.options');

        $variations = $product->variations->map(function ($variation) {
            return [
                'id' => $variation->id,
                'sku' => $variation->sku,
                'mrp' => $variation->mrp,
                'sp' => $variation->sp,
                'reseller_price' => $variation->reseller_price,
                'stock' => $variation->stock,
                'image_path' => $variation->image_path,
                // Construct a name from options for display
                'name' => $variation->options->map(function ($opt) {
                    return $opt->attribute_value;
                })->join(' / ') ?: 'Default',
            ];
        });

        return response()->json($variations);
    }

    public function updateQuickVariations(Request $request, Product $product)
    {
        $request->validate([
            'variations' => 'required|array',
            'variations.*.id' => 'required|exists:product_variations,id',
            'variations.*.mrp' => 'required|numeric',
            'variations.*.sp' => 'required|numeric',
            'variations.*.reseller_price' => 'nullable|numeric',
            'variations.*.stock' => 'required|integer',
            'copy_image_all' => 'nullable|boolean',
        ]);

        $copyImageAll = $request->boolean('copy_image_all');
        $masterImagePaths = null;

        // If copy all is enabled, check if the first variation has a file or we use its existing image
        if ($copyImageAll) {
            // Check for new upload on first variation
            if ($request->hasFile('variations.0.image')) {
                $file = $request->file('variations.0.image');
                $masterImagePaths = $this->processImage($file, 'products/variations');
            } elseif (isset($request->variations[0]['id'])) {
                // If no new file, checked valid existing image of first var?
                // Actually, if we just want to copy the *existing* image of var 0 to others,
                // we should fetch var 0 from DB.
                $firstVar = \App\Models\ProductVariation::find($request->variations[0]['id']);
                if ($firstVar && $firstVar->image_path) {
                    // Use existing paths. Note: processImage returns absolute paths usually in this controller
                    // but DB stores relevant relative paths often or full URLs. 
                    // Let's reuse what's in DB.
                    // However, processImage returns keys: 'original', 'thumbnail'
                    // We need to match that structure or handle it.
                    // Let's just grab the values directly.
                    $masterImagePaths = [
                        'original' => $firstVar->getRawOriginal('image_path'), // get raw to avoid accessor if needed, or just $firstVar->image_path? 
                        // The model accessor adds storage/ or asset(). We need the raw value to save back or consistent format.
                        // Actually, looking at store/update, we save what processImage returns ('/storage/path/to/img').
                        // So getting raw value is safer if we want to copy exactly.
                        // But wait, getImagePathAttribute adds storage/ prefix if missing. 
                        // Let's just use the logic: if we have paths, we update. 
                        'thumbnail' => $firstVar->getRawOriginal('thumbnail_path'),
                    ];
                }
            }
        }

        foreach ($request->variations as $index => $data) {
            $variation = \App\Models\ProductVariation::findOrFail($data['id']);

            $updateData = [
                'mrp' => $data['mrp'],
                'sp' => $data['sp'],
                'reseller_price' => $data['reseller_price'],
                'stock' => $data['stock'],
            ];

            // Image Logic
            if ($copyImageAll) {
                // Apply master image paths if available
                if ($masterImagePaths) {
                    $updateData['image_path'] = $masterImagePaths['original'];
                    $updateData['thumbnail_path'] = $masterImagePaths['thumbnail'];
                }
            } else {
                // Individual upload
                if ($request->hasFile("variations.{$index}.image")) {
                    $file = $request->file("variations.{$index}.image");
                    $paths = $this->processImage($file, 'products/variations');
                    $updateData['image_path'] = $paths['original'];
                    $updateData['thumbnail_path'] = $paths['thumbnail'];
                }
            }

            $variation->update($updateData);
        }

        return response()->json(['success' => true, 'message' => 'Variations updated successfully.']);
    }

    public function destroy(Product $product)
    {
        // Check if product has orders
        if ($product->orderItems()->exists()) {
            return redirect()->back()->with('error', 'Product cannot be deleted because it has associated orders.');
        }

        // Delete images from storage
        foreach ($product->images as $productImage) {
            $path = str_replace('/storage/', '', $productImage->image_path);
            if (\Illuminate\Support\Facades\Storage::disk('public')->exists($path)) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($path);
            }
            if ($productImage->thumbnail_path) {
                $thumbnailPath = str_replace('/storage/', '', $productImage->thumbnail_path);
                if (\Illuminate\Support\Facades\Storage::disk('public')->exists($thumbnailPath)) {
                    \Illuminate\Support\Facades\Storage::disk('public')->delete($thumbnailPath);
                }
            }
            $productImage->delete();
        }

        // Delete Variations
        $product->variations()->delete();

        // Delete the product
        $product->delete();

        return redirect()->route('admin.products.index')->with('success', 'Product deleted successfully');
    }
}

