<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\OrderReturn;
use App\Models\Order;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class OrderReturnController extends Controller
{
    public function index(Request $request)
    {
        $query = OrderReturn::with('order', 'user')->latest();

        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        $returns = $query->paginate(20);
        return view('admin.returns.index', compact('returns'));
    }

    public function show(OrderReturn $return)
    {
        $return->load('order.items.product', 'user');
        return view('admin.returns.show', compact('return'));
    }

    public function updateStatus(Request $request, OrderReturn $return)
    {
        $request->validate([
            'status' => 'required|in:approved,rejected,received,refund_initiated,refunded',
            'admin_note' => 'nullable|string',
        ]);

        $previousStatus = $return->status;
        $return->status = $request->status;
        if ($request->admin_note) {
            $return->admin_note = $request->admin_note;
        }

        // Logic for "received" - Restock?
        if ($request->status === 'received' && $previousStatus !== 'received') {
            $this->restockOrder($return->order);
        }

        // Logic for "refund_initiated" - Mark order as returned, update revenue
        if (($request->status === 'refund_initiated' || $request->status === 'refunded') && $return->order->status !== 'returned') {
            $return->order->update(['status' => 'returned']);
        }

        if ($request->status === 'refunded') {
            $return->refunded_at = now();
        }

        $return->save();

        return back()->with('success', 'Return status updated successfully.');
    }

    public function processRefund(Request $request, OrderReturn $return)
    {
        $request->validate([
            'deduction_amount' => 'nullable|numeric|min:0',
            'deduction_reason' => 'nullable|required_with:deduction_amount|string',
            'refund_proof' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
        ]);

        $totalPaid = $return->order->total; // Assuming total is stored in order
        $deduction = $request->deduction_amount ?? 0;
        $refundAmount = max(0, $totalPaid - $deduction);

        $return->update([
            'status' => 'refunded',
            'deduction_amount' => $deduction,
            'deduction_reason' => $request->deduction_reason,
            'refund_amount' => $refundAmount,
            'refunded_at' => now(),
        ]);

        if ($request->hasFile('refund_proof')) {
            $path = $request->file('refund_proof')->store('refund-proofs', 'public');
            $return->update(['refund_proof_path' => $path]);
        }

        // Update order status if not yet
        if ($return->order->status !== 'returned') {
            $return->order->update(['status' => 'returned']);
        }

        return back()->with('success', 'Refund processed successfully. Refund Amount: ' . $refundAmount);
    }

    private function restockOrder(Order $order)
    {
        foreach ($order->items as $item) {
            if ($item->variation_id) {
                $variation = \App\Models\ProductVariation::find($item->variation_id);
                if ($variation) {
                    $variation->increment('stock', $item->quantity);
                }
            } elseif ($item->product) {
                // If no variation, assumed simple product or main product stock management?
                // Usually simple products might track stock on 'products' table too?
                // The ProductController creates default variation for simple products.
                // Ideally we should always have a variation_id even for simple products?
                // Or we look for a default variation.
                // If item->variation_id is null, maybe we try to find the "default" variation of the product.
                $defaultVar = $item->product->variations()->first();
                if ($defaultVar) {
                    $defaultVar->increment('stock', $item->quantity);
                }
            }
        }
    }

    public function report(Request $request)
    {
        // Filter by month/year
        $query = OrderReturn::where('status', 'refunded')->with('order', 'user');

        if ($request->month && $request->year) {
            $query->whereMonth('refunded_at', $request->month)
                ->whereYear('refunded_at', $request->year);
        }

        $returns = $query->get();

        if ($request->action === 'pdf') {
            $pdf = Pdf::loadView('admin.reports.pdf.refunds', compact('returns', 'request'));
            return $pdf->download('refund-report.pdf');
        }

        return view('admin.reports.refunds', compact('returns'));
    }
}