<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Http\Request;

class OrderController extends Controller
{
    public function index(Request $request)
    {
        $status = $request->get('status');
        $search = $request->get('search');

        $query = Order::with(['user', 'agentRating', 'productReviews'])->latest();

        if ($status) {
            $query->where('status', $status);
        } else {
            // By default, maybe exclude pending_payment? Or show all.
            // Usually admins want to see confirmed orders first.
            // Let's show all except pending_payment strictly unless asked, or just show all.
            // Generally pending_payment are abandoned carts essentially.
            // We'll exclude them unless specifically asked for?
            // For now, let's exclude 'pending_payment' by default to avoid clutter.
            $query->where('status', '!=', 'pending_payment');
        }

        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('order_number', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%")
                    ->orWhereHas('user', function ($uq) use ($search) {
                        $uq->where('name', 'like', "%{$search}%")
                            ->orWhere('email', 'like', "%{$search}%");
                    });
            });
        }

        if ($request->has('role') && $request->role === 'Reseller') {
            $query->whereHas('user', function ($q) {
                $q->whereHas('roles', function ($rq) {
                    $rq->where('name', 'Reseller');
                });
            });
        } else {
            // Default: Show Client/Guest Orders ONLY (Exclude Resellers)
            $query->where(function ($subQ) {
                $subQ->whereNull('user_id')
                    ->orWhereHas('user', function ($uq) {
                        $uq->whereDoesntHave('roles', function ($rq) {
                            $rq->where('name', 'Reseller');
                        });
                    });
            });
        }

        if ($request->has('assigned')) {
            if ($request->boolean('assigned')) {
                $query->whereNotNull('delivery_agent_id');
            } else {
                $query->whereNull('delivery_agent_id');
            }
        }

        if ($request->filled('delivery_agent_id')) {
            $query->where('delivery_agent_id', $request->delivery_agent_id);
        }

        if ($request->start_date) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
        if ($request->end_date) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }

        $orders = $query->paginate(20);

        // Calculate counts for admin tabs/filters if we want them there (though admin view currently uses simple links)
        // Let's pass them just in case or for future use, but the user requested for admin as well.
        // Wait, the user said "implement count logic in admin as well" which likely means on the tabs in admin/orders/index.blade.php
        // Helper to get base query for counts
        $baseCountQuery = clone $query;
        // Reset order/limit/offset from the main query but keep filters
        $baseCountQuery->getQuery()->orders = null;
        $baseCountQuery->getQuery()->limit = null;
        $baseCountQuery->getQuery()->offset = null;
        // We need to remove the status constraint ONLY for the status specific counts
        // BUT the main $query ALREADY has status applied if $request->status is present.
        // This makes it hard to clone $query for counts of OTHER statuses.

        // Better approach: Re-build the base context query (role, search, dates)
        $countContext = Order::query();
        if ($request->has('role') && $request->role === 'Reseller') {
            $countContext->whereHas('user', function ($q) {
                $q->whereHas('roles', function ($rq) {
                    $rq->where('name', 'Reseller');
                });
            });
        } else {
            // Default Context: Exclude Resellers
            $countContext->where(function ($subQ) {
                $subQ->whereNull('user_id')
                    ->orWhereHas('user', function ($uq) {
                        $uq->whereDoesntHave('roles', function ($rq) {
                            $rq->where('name', 'Reseller');
                        });
                    });
            });
        }
        // Apply default exclusion
        // $countContext->where('status', '!=', 'pending_payment'); // Keep consistency

        $counts = [
            'all' => (clone $countContext)->where('status', '!=', 'pending_payment')->count(),
            'pending' => (clone $countContext)->where('status', 'pending')->count(),
            'processing' => (clone $countContext)->where('status', 'processing')->count(),
            'shipped' => (clone $countContext)->where('status', 'shipped')->count(),
            'completed' => (clone $countContext)->where('status', 'completed')->count(),
            'cancelled' => (clone $countContext)->where('status', 'cancelled')->count(),
            'returned' => (clone $countContext)->where('status', 'returned')->count(),
        ];

        return view('admin.orders.index', compact('orders', 'counts'));
    }

    public function show(Order $order)
    {
        $order->load(['items.product', 'items.variation', 'shippingAddress', 'billingAddress', 'user', 'deliveryAgent.user', 'statusHistories', 'pickupStore', 'productReviews.product', 'agentRating']);
        $deliveryAgents = \App\Models\DeliveryAgent::where('status', true)->where('is_available', true)->with('user')->get();
        $stores = \App\Models\Store::where('is_active', true)->get();
        return view('admin.orders.show', compact('order', 'deliveryAgents', 'stores'));
    }

    public function downloadInvoice(Order $order)
    {
        $order->load(['items.product', 'items.variation', 'shippingAddress', 'billingAddress', 'user']);

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('pdf.invoice', compact('order'));
        return $pdf->download('invoice-' . $order->order_number . '.pdf');
    }

    public function updateStatus(Request $request, Order $order)
    {
        $request->validate([
            'status' => 'required|in:pending,processing,shipped,completed,cancelled'
        ]);

        $previousStatus = $order->status;
        $order->update(['status' => $request->status]);

        // Wallet Logic
        $user = $order->user;
        if ($user) {
            $settings = \App\Models\GeneralSetting::firstOrNew();

            if ($settings->wallet_enabled) {
                // 1. Handle Completion (Award Points)
                // 1. Handle Completion (Award Points)
                if ($request->status == 'completed' && $order->payment_status != 'refunded') {
                    // Check if points already earned
                    $alreadyEarned = $user->walletTransactions()
                        ->where('reference_type', 'App\Models\Order')
                        ->where('reference_id', $order->id)
                        ->where('type', 'credit')
                        ->where('description', 'like', 'Earned%')
                        ->exists();

                    if (!$alreadyEarned && $settings->wallet_earn_rate > 0) {
                        $earnPoints = floor(($order->total / 100) * $settings->wallet_earn_rate);
                        if ($earnPoints > 0) {
                            $user->creditWallet($earnPoints, "Earned from Order #" . $order->order_number, 'App\Models\Order', $order->id);
                        }
                    }

                    // Also ensure payment status is paid for COD orders upon completion
                    if ($order->payment_status == 'pending') {
                        $order->update(['payment_status' => 'paid']);
                    }
                }

                // 2. Handle Cancellation (Refund Redeemed Points & Revert Earned Points)
                if ($request->status == 'cancelled') {
                    // Revert Redeemed Points (Refund to user)
                    if ($order->points_redeemed > 0) {
                        // Check if already refunded
                        $alreadyRefunded = $user->walletTransactions()
                            ->where('reference_type', 'App\Models\Order')
                            ->where('reference_id', $order->id)
                            ->where('description', 'like', 'Refund%')
                            ->exists();

                        if (!$alreadyRefunded) {
                            $user->creditWallet($order->points_redeemed, "Refund for cancelled Order #" . $order->order_number, 'App\Models\Order', $order->id);
                        }
                    }

                    // Gift Card Refund Logic
                    if ($order->gift_card_discount > 0) {
                        // Check if already refunded
                        $alreadyGcRefunded = $user->walletTransactions()
                            ->where('reference_type', 'App\Models\Order')
                            ->where('reference_id', $order->id)
                            ->where('description', 'like', 'Gift Card Refund%')
                            ->exists();

                        if (!$alreadyGcRefunded) {
                            $user->creditWallet($order->gift_card_discount, "Gift Card Refund for cancelled Order #" . $order->order_number, 'App\Models\Order', $order->id);
                        }
                    }

                    // Revert Earned Points (Debit from user)
                    // If user was awarded points, take them back.
                    $earnedTransaction = $user->walletTransactions()
                        ->where('reference_type', 'App\Models\Order')
                        ->where('reference_id', $order->id)
                        ->where('type', 'credit')
                        ->where('description', 'like', 'Earned%')
                        ->first();

                    if ($earnedTransaction) {
                        $alreadyReversed = $user->walletTransactions()
                            ->where('reference_type', 'App\Models\Order')
                            ->where('reference_id', $order->id)
                            ->where('type', 'debit')
                            ->where('description', 'like', 'Reversal%')
                            ->exists();

                        if (!$alreadyReversed) {
                            if ($user->wallet_balance >= $earnedTransaction->amount) {
                                $user->debitWallet($earnedTransaction->amount, "Reversal of points from Order #" . $order->order_number, 'App\Models\Order', $order->id);
                            } else {
                                // Handle insufficient funds: Debit whatever is left
                                if ($user->wallet_balance > 0) {
                                    $user->debitWallet($user->wallet_balance, "Reversal (Partial) from Order #" . $order->order_number, 'App\Models\Order', $order->id);
                                }
                            }
                        }
                    }
                }
            }
        }

        // Email Triggers
        if ($request->status == 'shipped' && $previousStatus != 'shipped') {
            \App\Services\OrderMailService::sendNotification($order, 'Order Shipped', 'Clients');
        }

        if ($request->status == 'completed' && $previousStatus != 'completed') {
            \App\Services\OrderMailService::sendNotification($order, 'Completed', 'Clients');
            \App\Services\OrderMailService::sendNotification($order, 'Order Delivered', 'Admin');
        }

        if ($request->status == 'cancelled' && $previousStatus != 'cancelled') {
            \App\Services\OrderMailService::sendNotification($order, 'Cancelled', 'Admin');
        }

        return back()->with('success', 'Order status updated successfully.');
    }

    public function assignAgent(Request $request, Order $order)
    {
        if ($order->status == 'cancelled' || $order->status == 'completed' || $order->status == 'returned') {
            return back()->with('error', 'Cannot assign agent to a ' . $order->status . ' order.');
        }

        $request->validate([
            'delivery_agent_id' => 'required|exists:delivery_agents,id',
            'pickup_store_id' => 'nullable|exists:stores,id',
        ]);

        $updateData = ['delivery_agent_id' => $request->delivery_agent_id];

        // Only allow pickup store assignment for local and pending orders
        if ($order->order_scope === 'Local' && $order->status === 'pending') {
            $updateData['pickup_store_id'] = $request->pickup_store_id;
        }

        $order->update($updateData);

        // Send FCM Notification to the assigned agent
        $agent = \App\Models\DeliveryAgent::with('user')->find($request->delivery_agent_id);
        if ($agent && $agent->user && $agent->user->fcm_token) {
            $fcmService = new \App\Services\FCMService();
            $title = 'New Order Assigned';
            $body = 'Order #' . $order->order_number . ' has been assigned to you.';
            $data = [
                'type' => 'order_assigned',
                'order_id' => $order->id,
            ];
            $fcmService->sendNotification($agent->user->fcm_token, $title, $body, $data);
        }

        return back()->with('success', 'Delivery assigned successfully.');
    }

    public function generateQuotation(Request $request, Order $order)
    {
        $request->validate([
            'items' => 'required|array',
            'items.*.price' => 'required|numeric|min:0',
            'items.*.discount_percent' => 'nullable|numeric|min:0|max:100',
            'overall_discount_percent' => 'nullable|numeric|min:0|max:100',
        ]);

        $subtotal = 0;

        // Calculate Overall Discount Factor
        $overallDiscountPercent = $request->overall_discount_percent ?? 0;
        $overallDiscountFactor = 1 - ($overallDiscountPercent / 100);

        foreach ($request->items as $itemId => $data) {
            $item = $order->items()->find($itemId);
            if ($item) {
                // Update Base Price first
                $price = $data['price'];
                $itemDiscountPercent = $data['discount_percent'] ?? 0;

                // Calculate Effective Discount Percentage combining Item & Overall
                // Formula: 1 - ((1 - Item%) * (1 - Overall%))
                // Example: 10% Item + 10% Overall = 1 - (0.9 * 0.9) = 1 - 0.81 = 0.19 = 19%
                $combinedDiscountPercent = 100 * (1 - ((1 - ($itemDiscountPercent / 100)) * $overallDiscountFactor));

                $options = $item->options ?? [];
                $options['discount_percentage'] = number_format($combinedDiscountPercent, 2); // Store effective discount

                // Calculate Line Total using Combined Discount
                $finalUnitTestPrice = $price * (1 - ($combinedDiscountPercent / 100));
                $lineTotal = $finalUnitTestPrice * $item->quantity;

                $item->price = $price; // Store Base Price (undiscounted)
                $item->total = $lineTotal; // Final Line Total (after all discounts)
                $item->options = $options;
                $item->save();

                $subtotal += $lineTotal;
            }
        }

        // Update Order Totals
        // Subtotal is now the Sum of (Discounted) Lines.
        // Discount column is 0 because discounts are baked into lines.
        $order->subtotal = $subtotal;
        $order->discount = 0;
        $order->total = $subtotal + $order->shipping_cost;
        $order->status = 'quotation_sent';
        $order->save();

        // Send Email
        if ($order->email) {
            try {
                \Illuminate\Support\Facades\Mail::to($order->email)->send(new \App\Mail\ResellerQuotation($order));
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::error("Failed to send quotation email: " . $e->getMessage());
            }
        }

        return back()->with('success', 'Quotation generated and sent to reseller.');
    }
}
