<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderItem;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class LeadershipBoardController extends Controller
{
    public function index(Request $request)
    {
        // Filter Logic
        $year = $request->input('year', Carbon::now()->year);
        $month = $request->input('month', Carbon::now()->month);

        $startDate = Carbon::createFromDate($year, $month, 1)->startOfMonth();
        $endDate = $startDate->copy()->endOfMonth();

        // Adjust for future dates if current month
        if ($endDate->isFuture()) {
            $endDate = Carbon::now();
        }
        // If the selected month is completely in the future, show empty or stop? 
        // Better to just let it show empty 0s.
        // But let's reset to full end of month for historical data consistency, 
        // except charts might look flat at the end. 
        // Let's stick to end of month unless it's strictly current month, then 'today' is fine but for "Revenue Table" we show full month usually.
        $endDate = Carbon::createFromDate($year, $month, 1)->endOfMonth();


        // Previous Year Comparison (Same period last year)
        // Logic: Same month last year
        $prevStartDate = $startDate->copy()->subYear();
        $prevEndDate = $endDate->copy()->subYear();

        // Calculate Stats (Top Cards)
        // These stats should probably respect the filter too? 
        // The prompt says "add month and year filter". 
        // Usually top cards are for the filtered range.

        $stats = [
            'total_sales' => $this->getTotalSales($startDate, $endDate),
            'net_sales' => $this->getNetSales($startDate, $endDate),
            'orders_count' => $this->getOrdersCount($startDate, $endDate),
            'products_sold' => $this->getProductsSold($startDate, $endDate),
            'variations_sold' => $this->getVariationsSold($startDate, $endDate),
        ];

        $prevStats = [
            'total_sales' => $this->getTotalSales($prevStartDate, $prevEndDate),
            'net_sales' => $this->getNetSales($prevStartDate, $prevEndDate),
            'orders_count' => $this->getOrdersCount($prevStartDate, $prevEndDate),
            'products_sold' => $this->getProductsSold($prevStartDate, $prevEndDate),
            'variations_sold' => $this->getVariationsSold($prevStartDate, $prevEndDate),
        ];

        // Changes
        $changes = [];
        foreach ($stats as $key => $value) {
            $prevValue = $prevStats[$key];
            if ($prevValue > 0) {
                $changes[$key] = round((($value - $prevValue) / $prevValue) * 100);
            } else {
                $changes[$key] = $value > 0 ? 100 : 0;
            }
        }

        // Charts Data
        $dailySales = $this->getDailyChartData($startDate, $endDate, 'total_amount');
        $dailyOrders = $this->getDailyChartData($startDate, $endDate, 'id', true); // true for count

        // Revenue Table Data
        $revenueData = $this->getRevenueTableData($startDate, $endDate);

        // Filter Options
        $years = range(Carbon::now()->year, Carbon::now()->subYears(4)->year);
        $months = [];
        for ($m = 1; $m <= 12; $m++) {
            $months[$m] = Carbon::create()->month($m)->format('F');
        }

        return view('admin.leadership_board.index', compact(
            'stats',
            'changes',
            'dailySales',
            'dailyOrders',
            'revenueData',
            'startDate',
            'endDate',
            'year',
            'month',
            'years',
            'months'
        ));
    }

    public function downloadPdf(Request $request)
    {
        $year = $request->input('year', Carbon::now()->year);
        $month = $request->input('month', Carbon::now()->month);

        $startDate = Carbon::createFromDate($year, $month, 1)->startOfMonth();
        $endDate = $startDate->copy()->endOfMonth();

        $revenueData = $this->getRevenueTableData($startDate, $endDate);

        $pdf = \PDF::loadView('admin.leadership_board.pdf', compact('revenueData', 'startDate', 'endDate'));
        return $pdf->download("revenue-report-{$year}-{$month}.pdf");
    }

    private function getRevenueTableData($startDate, $endDate)
    {
        // Initialize all days in the month
        $dates = [];
        $period = \Carbon\CarbonPeriod::create($startDate, $endDate);
        foreach ($period as $date) {
            $dates[$date->format('Y-m-d')] = [
                'date' => $date,
                'orders' => 0,
                'gross_sales' => 0,
                'returns' => 0,
                'coupons' => 0,
                'net_sales' => 0,
                'taxes' => 0,
                'shipping' => 0,
                'total_sales' => 0,
            ];
        }

        // Fetch Order Data
        // Status: excluding pending_payment, failed. 
        // Note: For "Returns" we might check 'cancelled' status but for "Sales" we exclude it?
        // Let's fetch valid orders first.
        // Fetch valid orders first.
        $orders = Order::whereBetween('created_at', [$startDate, $endDate])
            ->get();

        // Check Shipping Tax Settings
        $settings = \App\Models\GeneralSetting::first();
        $shippingTaxRate = 0;
        $shippingTaxEnabled = false;

        if ($settings && $settings->shipping_tax_enabled && $settings->shipping_tax_rate_id) {
            $shippingTaxEnabled = true;
            $rateObj = \App\Models\TaxRate::find($settings->shipping_tax_rate_id);
            if ($rateObj) {
                $shippingTaxRate = $rateObj->rate;
            }
        }

        foreach ($orders as $order) {
            $d = $order->created_at->format('Y-m-d');
            if (!isset($dates[$d]))
                continue;

            // Logic:
            // If status is cancelled -> count as Return? 
            // Or just ignore from sales?
            // Screenshot has "Returns" column.

            if ($order->status === 'cancelled' || $order->status === 'returned') {
                $dates[$d]['returns'] += $order->total_amount; // Assuming full return
                // Do we count it in Orders? Maybe.
                // Do we count it in Gross Sales? 
                // Gross Sales usually includes everything, then Returns is a deduction.
                // So yes:
                $dates[$d]['gross_sales'] += $order->subtotal;
                $dates[$d]['orders'] += 1;
                // Net Sales = Gross - Returns - Coupons
                // Note: If order is cancelled, usually Net Sales for it is 0.
                // My formula: Net = Gross - Returns - Coupons.
                // If Gross=100, Coupon=0, Return=100. Net = 0. Correct.

                $dates[$d]['coupons'] += $order->discount;
                // Tax? Shipping?
                // If returned, tax/shipping might be refunded too. 
                // Let's execute logic below.
            } else if (in_array($order->status, ['pending_payment', 'failed'])) {
                // Ignore these usually
                continue;
            } else {
                // Valid Orders
                $dates[$d]['orders'] += 1;
                $dates[$d]['gross_sales'] += $order->subtotal;
                $dates[$d]['coupons'] += $order->discount;
                $dates[$d]['shipping'] += $order->shipping_cost;

                // Calculate Tax
                $orderTax = 0;
                foreach ($order->items as $item) {
                    // Determine if product tax is enabled
                    $storedTax = (float) $item->tax_amount;
                    if ($storedTax > 0) {
                        $orderTax += $storedTax;
                    } elseif ($settings && $settings->product_tax_enabled) {
                        // Fallback logic
                        $rate = $item->product ? ($item->product->tax_code ?? 18) : 18;
                        $itemTotal = $item->price * $item->quantity;
                        $taxComponent = $itemTotal * ($rate / (100 + $rate));
                        $orderTax += $taxComponent;
                    }
                    // Else: tax is 0 (already initialized to 0)
                }

                // Add Shipping Tax (Stored)
                $shippingTax = $order->shipping_tax_amount;
                if ($shippingTax > 0) {
                    $orderTax += $shippingTax;
                    // Adjust Shipping Net
                    $dates[$d]['shipping'] -= $shippingTax;
                }

                $dates[$d]['taxes'] += $orderTax;

                // Total Sales
                $dates[$d]['total_sales'] += $order->total_amount; // or subtotal + shipping - discount
            }
        }

        // Calculate Final columns
        foreach ($dates as &$row) {
            // Net Sales = Total Sales - Tax - Shipping.
            // (Returns are already handled by not adding to valid sales)

            $row['net_sales'] = $row['total_sales'] - $row['taxes'] - $row['shipping'];
        }

        // Sort descending by date
        krsort($dates);

        return $dates;
    }

    private function getBaseOrderQuery($startDate, $endDate)
    {
        return Order::whereBetween('created_at', [$startDate, $endDate])
            ->whereNotIn('status', ['cancelled', 'pending_payment', 'failed', 'returned']);
    }

    private function getTotalSales($startDate, $endDate)
    {
        return $this->getBaseOrderQuery($startDate, $endDate)->sum('total_amount');
    }

    private function getNetSales($startDate, $endDate)
    {
        return $this->getBaseOrderQuery($startDate, $endDate)->sum('total_amount');
    }

    private function getOrdersCount($startDate, $endDate)
    {
        return $this->getBaseOrderQuery($startDate, $endDate)->count();
    }

    private function getProductsSold($startDate, $endDate)
    {
        return OrderItem::whereHas('order', function ($q) use ($startDate, $endDate) {
            $q->whereBetween('created_at', [$startDate, $endDate])
                ->whereNotIn('status', ['cancelled', 'pending_payment', 'failed', 'returned']);
        })->sum('quantity');
    }

    private function getVariationsSold($startDate, $endDate)
    {
        return OrderItem::whereHas('order', function ($q) use ($startDate, $endDate) {
            $q->whereBetween('created_at', [$startDate, $endDate])
                ->whereNotIn('status', ['cancelled', 'pending_payment', 'failed', 'returned']);
        })->whereNotNull('options')->sum('quantity');
    }

    private function getDailyChartData($startDate, $endDate, $column, $isCount = false)
    {
        $query = Order::whereBetween('created_at', [$startDate, $endDate])
            ->whereNotIn('status', ['cancelled', 'pending_payment', 'failed', 'returned']);

        if ($isCount) {
            $data = $query->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as value'));
        } else {
            $data = $query->select(DB::raw('DATE(created_at) as date'), DB::raw("SUM($column) as value"));
        }

        $result = $data->groupBy('date')
            ->orderBy('date')
            ->get()
            ->pluck('value', 'date')
            ->toArray();

        // Fill missing dates with 0
        $filledData = [];
        $period = \Carbon\CarbonPeriod::create($startDate, $endDate);
        foreach ($period as $date) {
            $d = $date->format('Y-m-d');
            $filledData[$d] = $result[$d] ?? 0;
        }
        return $filledData;
    }
}
