<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\EmailTemplate;
use Illuminate\Http\Request;

class EmailTemplateController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $templates = EmailTemplate::latest()->paginate(10);
        return view('admin.email-templates.index', compact('templates'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = [
            'New Arrivals',
            'For Discounts',
            'Pending Orders',
            'Order Shipped',
            'Completed',
            'Cancelled'
        ];
        return view('admin.email-templates.create', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'subject' => 'required|string|max:255',
            'category' => 'required|string',
            'target_role' => 'required|string|in:Admin,Agents,Clients,Resellers',
            'custom_category' => 'nullable|string|max:255',
            'content' => 'required|string',
        ]);

        $category = $request->category === 'other' ? $request->custom_category : $request->category;

        EmailTemplate::create([
            'name' => $request->name,
            'subject' => $request->subject,
            'category' => $category,
            'target_role' => $request->target_role,
            'content' => $request->input('content'),
            'is_active' => $request->has('is_active'),
        ]);

        return redirect()->route('admin.email-templates.index')->with('success', 'Email template created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(EmailTemplate $emailTemplate)
    {
        $categories = [
            'New Arrivals',
            'For Discounts',
            'Pending Orders',
            'Order Shipped',
            'Completed',
            'Cancelled'
        ];
        return view('admin.email-templates.edit', compact('emailTemplate', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, EmailTemplate $emailTemplate)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'subject' => 'required|string|max:255',
            'category' => 'required|string',
            'target_role' => 'required|string|in:Admin,Agents,Clients,Resellers',
            'custom_category' => 'nullable|string|max:255',
            'content' => 'required|string',
        ]);

        $category = $request->category === 'other' ? $request->custom_category : $request->category;

        $emailTemplate->update([
            'name' => $request->name,
            'subject' => $request->subject,
            'category' => $category,
            'target_role' => $request->target_role,
            'content' => $request->input('content'),
            'is_active' => $request->has('is_active'),
        ]);

        return redirect()->route('admin.email-templates.index')->with('success', 'Email template updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function show(EmailTemplate $emailTemplate)
    {
        return view('admin.email-templates.show', compact('emailTemplate'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(EmailTemplate $emailTemplate)
    {
        $emailTemplate->delete();
        return redirect()->route('admin.email-templates.index')->with('success', 'Email template deleted successfully.');
    }
}
