<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Discount;
use App\Models\Product;
use App\Models\Category;

class DiscountController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $discounts = Discount::latest()->paginate(10);
        return view('admin.discounts.index', compact('discounts'));
    }

    public function create()
    {
        $products = Product::select('id', 'name')->get();
        $categories = Category::select('id', 'name')->get();
        return view('admin.discounts.create', compact('products', 'categories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:fixed,percentage,buy_x_get_y',
            'value' => 'required_unless:type,buy_x_get_y|nullable|numeric',
            'buy_x' => 'required_if:type,buy_x_get_y|nullable|integer|min:1',
            'get_y' => 'required_if:type,buy_x_get_y|nullable|integer|min:1',
            'valid_from' => 'nullable|date',
            'valid_until' => 'nullable|date|after_or_equal:valid_from',
            'apply_to' => 'required|in:products,categories',
            'products' => 'required_if:apply_to,products|array',
            'categories' => 'required_if:apply_to,categories|array',
        ]);

        $discount = Discount::create($request->only([
            'name',
            'type',
            'value',
            'buy_x',
            'get_y',
            'valid_from',
            'valid_until',
            'is_active'
        ]));

        if ($request->apply_to === 'products') {
            $discount->products()->sync($request->products);
        } elseif ($request->apply_to === 'categories') {
            $discount->categories()->sync($request->categories);
        }

        return redirect()->route('admin.discounts.index')->with('success', 'Discount created successfully.');
    }

    public function edit(Discount $discount)
    {
        $products = Product::select('id', 'name')->get();
        $categories = Category::select('id', 'name')->get();
        return view('admin.discounts.edit', compact('discount', 'products', 'categories'));
    }

    public function update(Request $request, Discount $discount)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:fixed,percentage,buy_x_get_y',
            'value' => 'required_unless:type,buy_x_get_y|nullable|numeric',
            'buy_x' => 'required_if:type,buy_x_get_y|nullable|integer|min:1',
            'get_y' => 'required_if:type,buy_x_get_y|nullable|integer|min:1',
            'valid_from' => 'nullable|date',
            'valid_until' => 'nullable|date|after_or_equal:valid_from',
            'apply_to' => 'required|in:products,categories',
            'products' => 'required_if:apply_to,products|array',
            'categories' => 'required_if:apply_to,categories|array',
        ]);

        $discount->update($request->only([
            'name',
            'type',
            'value',
            'buy_x',
            'get_y',
            'valid_from',
            'valid_until',
            'is_active'
        ]));

        if ($request->apply_to === 'products') {
            $discount->categories()->detach();
            $discount->products()->sync($request->products);
        } elseif ($request->apply_to === 'categories') {
            $discount->products()->detach();
            $discount->categories()->sync($request->categories);
        }

        return redirect()->route('admin.discounts.index')->with('success', 'Discount updated successfully.');
    }

    public function destroy(Discount $discount)
    {
        $discount->delete();
        return redirect()->route('admin.discounts.index')->with('success', 'Discount deleted successfully.');
    }

    public function toggleStatus(Discount $discount)
    {
        $discount->update(['is_active' => !$discount->is_active]);
        return back()->with('success', 'Discount status updated successfully.');
    }
}
