<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Process;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Symfony\Component\HttpFoundation\StreamedResponse;

class DeveloperSettingController extends Controller
{
    public function index()
    {
        $seedersPath = database_path('seeders');
        $seeders = [];
        if (file_exists($seedersPath)) {
            $files = scandir($seedersPath);
            foreach ($files as $file) {
                if (str_ends_with($file, '.php') && $file !== 'DatabaseSeeder.php') {
                    $seeders[] = str_replace('.php', '', $file);
                }
            }
        }
        return view('admin.settings.developer', compact('seeders'));
    }

    public function clearCache()
    {
        try {
            Artisan::call('optimize:clear');
            return back()->with('success', 'Cache cleared successfully!');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to clear cache: ' . $e->getMessage());
        }
    }

    public function runBuild()
    {
        try {
            // Fixing permission just in case
            Process::run('chmod +x base_path("node_modules/.bin/vite")');

            $result = Process::path(base_path())->run('npm run build');

            if ($result->successful()) {
                return back()->with('success', 'Build executed successfully! Output: ' . $result->output());
            } else {
                return back()->with('error', 'Build failed! Error: ' . $result->errorOutput());
            }
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to run build: ' . $e->getMessage());
        }
    }

    public function exportDatabase()
    {
        try {
            $filename = 'backup-' . date('Y-m-d-H-i-s') . '.sql';
            $path = storage_path('app/backups/' . $filename);

            // Ensure directory exists
            if (!file_exists(storage_path('app/backups'))) {
                mkdir(storage_path('app/backups'), 0755, true);
            }

            // Get database configuration
            $dbHost = config('database.connections.mysql.host');
            $dbName = config('database.connections.mysql.database');
            $dbUser = config('database.connections.mysql.username');
            $dbPass = config('database.connections.mysql.password');

            // Construct mysqldump command
            // Note: relying on mysqldump being in the path
            $command = "mysqldump --user={$dbUser} --password={$dbPass} --host={$dbHost} {$dbName} > {$path}";

            $result = Process::run($command);

            if ($result->successful()) {
                if (file_exists($path)) {
                    return response()->download($path)->deleteFileAfterSend(true);
                } else {
                    return back()->with('error', 'Backup file was not created.');
                }
            } else {
                return back()->with('error', 'Database export failed. Error: ' . $result->errorOutput());
            }

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to export database: ' . $e->getMessage());
        }
    }

    public function importDatabase(Request $request)
    {
        $request->validate([
            'sql_file' => 'required|file|mimes:sql,txt',
        ]);

        try {
            $file = $request->file('sql_file');
            $path = $file->getRealPath();

            // Get database configuration
            $dbHost = config('database.connections.mysql.host');
            $dbName = config('database.connections.mysql.database');
            $dbUser = config('database.connections.mysql.username');
            $dbPass = config('database.connections.mysql.password');

            $command = "mysql --user={$dbUser} --password={$dbPass} --host={$dbHost} {$dbName} < {$path}";

            $result = Process::run($command);

            if ($result->successful()) {
                return back()->with('success', 'Database imported successfully!');
            } else {
                return back()->with('error', 'Database import failed. Error: ' . $result->errorOutput());
            }

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to import database: ' . $e->getMessage());
        }
    }
    public function migrateDatabase()
    {
        try {
            Artisan::call('migrate', ['--force' => true]);
            return back()->with('success', 'Database migration executed successfully! Output: ' . Artisan::output());
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to migrate database: ' . $e->getMessage());
        }
    }

    public function runSeeder(Request $request)
    {
        $request->validate([
            'class' => 'nullable|string'
        ]);

        try {
            $params = ['--force' => true];
            if ($request->class) {
                $params['--class'] = $request->class;
            }

            Artisan::call('db:seed', $params);

            $message = $request->class ? "Seeder [{$request->class}]" : "All seeders";
            return back()->with('success', "{$message} executed successfully! Output: " . Artisan::output());
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to run seeder: ' . $e->getMessage());
        }
    }
}
