<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\DeliveryAgent;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class DeliveryAgentController extends Controller
{
    public function index()
    {
        $agents = DeliveryAgent::with(['user', 'vehicles', 'documents', 'ratings'])->latest()->get();
        return view('admin.delivery_agents.index', compact('agents'));
    }

    public function create()
    {
        return view('admin.delivery_agents.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            // User login info
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8',

            // Profile
            'profile_image' => 'nullable|image|max:2048',
            'phone' => 'required|string|max:20',
            'alternate_phone' => 'nullable|string|max:20',
            'dob' => 'nullable|date',
            'gender' => 'nullable|in:Male,Female,Other',
            'father_name' => 'nullable|string|max:255',
            'spouse_name' => 'nullable|string|max:255',
            'address' => 'required|string',

            // Multiple Documents (KYC/License)
            'documents' => 'required|array|min:1',
            'documents.*.type' => 'required|string',
            'documents.*.number' => 'nullable|string',
            'documents.*.image' => 'required|image|max:2048',

            // Multiple Vehicles
            'vehicles' => 'required|array|min:1',
            'vehicles.*.type' => 'required|string',
            'vehicles.*.vehicle_number' => 'required|string',
            'vehicles.*.model' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'is_active' => true,
            ]);

            // Assign Role
            $role = \App\Models\Role::firstOrCreate(['name' => 'delivery_agent']);
            $user->roles()->attach($role);

            $profilePath = $request->file('profile_image') ? $request->file('profile_image')->store('agents/profile', 'public') : null;

            $agent = DeliveryAgent::create([
                'user_id' => $user->id,
                'profile_image' => $profilePath,
                'phone' => $request->phone,
                'alternate_phone' => $request->alternate_phone,
                'dob' => $request->dob,
                'gender' => $request->gender,
                'father_name' => $request->father_name,
                'spouse_name' => $request->spouse_name,
                'address' => $request->address,
                'status' => true,
            ]);

            // Save Documents
            foreach ($request->documents as $doc) {
                if (isset($doc['image'])) {
                    $path = $doc['image']->store('agents/documents', 'public');
                    $agent->documents()->create([
                        'type' => $doc['type'],
                        'number' => $doc['number'] ?? null,
                        'image_path' => $path,
                    ]);
                }
            }

            // Save Vehicles
            foreach ($request->vehicles as $vehicle) {
                $agent->vehicles()->create([
                    'type' => $vehicle['type'],
                    'vehicle_number' => $vehicle['vehicle_number'],
                    'model' => $vehicle['model'] ?? null,
                    'is_primary' => false, // Can determine logic for primary later
                ]);
            }

            DB::commit();
            return redirect()->route('admin.delivery-agents.index')->with('success', 'Delivery Agent created successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error creating agent: ' . $e->getMessage())->withInput();
        }
    }

    public function show(DeliveryAgent $deliveryAgent)
    {
        $deliveryAgent->load([
            'user',
            'documents',
            'vehicles',
            'ratings' => function ($query) {
                $query->with(['user', 'order'])->latest();
            }
        ]);

        $cashCollected = \App\Models\Order::where('delivery_agent_id', $deliveryAgent->id)
            ->where('status', 'completed')
            ->where('payment_method', 'cod')
            ->sum('total');

        $codOrderCount = \App\Models\Order::where('delivery_agent_id', $deliveryAgent->id)
            ->where('status', 'completed')
            ->where('payment_method', 'cod')
            ->count();

        return view('admin.delivery_agents.show', compact('deliveryAgent', 'cashCollected', 'codOrderCount'));
    }

    public function edit(DeliveryAgent $deliveryAgent)
    {
        $deliveryAgent->load(['user', 'documents', 'vehicles']);
        return view('admin.delivery_agents.edit', compact('deliveryAgent'));
    }

    public function update(Request $request, DeliveryAgent $deliveryAgent)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $deliveryAgent->user_id,
            'phone' => 'required|string|max:20',
            'alternate_phone' => 'nullable|string|max:20',
            'dob' => 'nullable|date',
            'gender' => 'nullable|in:Male,Female,Other',
            'father_name' => 'nullable|string|max:255',
            'spouse_name' => 'nullable|string|max:255',
            'profile_image' => 'nullable|image|max:2048',
            'status' => 'nullable|boolean',

            // Existing Documents Validation
            'existing_documents' => 'nullable|array',
            'existing_documents.*.type' => 'required|string',
            'existing_documents.*.number' => 'nullable|string',
            'existing_documents.*.image' => 'nullable|image|max:2048',

            // Existing Vehicles
            'existing_vehicles' => 'nullable|array',
            'existing_vehicles.*.type' => 'required|string',
            'existing_vehicles.*.vehicle_number' => 'required|string',
            'existing_vehicles.*.model' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            // Update User
            $deliveryAgent->user->update([
                'name' => $request->name,
                'email' => $request->email,
            ]);

            if ($request->filled('password')) {
                $deliveryAgent->user->update([
                    'password' => Hash::make($request->password)
                ]);
            }

            // Update Agent Details
            $data = $request->except([
                'name',
                'email',
                'password',
                'profile_image',
                'kyc_image',
                'license_image', // Files/Legacy
                'documents',
                'vehicles', // New Relations
                'existing_documents',
                'existing_vehicles' // Existing Relations updates
            ]);

            if ($request->has('status')) {
                $data['status'] = $request->boolean('status');
            }

            if ($request->hasFile('profile_image')) {
                if ($deliveryAgent->profile_image)
                    Storage::disk('public')->delete($deliveryAgent->profile_image);
                $data['profile_image'] = $request->file('profile_image')->store('agents/profile', 'public');
            }

            // Update Existing Documents
            if ($request->has('existing_documents')) {
                foreach ($request->existing_documents as $id => $docData) {
                    $doc = \App\Models\AgentDocument::find($id);
                    if ($doc && $doc->delivery_agent_id == $deliveryAgent->id) {
                        $updateData = [
                            'type' => $docData['type'],
                            'number' => $docData['number'] ?? null,
                        ];
                        // Handle File Replace
                        if ($request->hasFile("existing_documents.{$id}.image")) {
                            $file = $request->file("existing_documents.{$id}.image");
                            if ($file) {
                                if ($doc->image_path)
                                    Storage::disk('public')->delete($doc->image_path);
                                $updateData['image_path'] = $file->store('agents/documents', 'public');
                            }
                        }
                        $doc->update($updateData);
                    }
                }
            }

            // Update Existing Vehicles
            if ($request->has('existing_vehicles')) {
                foreach ($request->existing_vehicles as $id => $vehicleData) {
                    $vehicle = \App\Models\AgentVehicle::find($id);
                    if ($vehicle && $vehicle->delivery_agent_id == $deliveryAgent->id) {
                        $vehicle->update([
                            'type' => $vehicleData['type'],
                            'vehicle_number' => $vehicleData['vehicle_number'],
                            'model' => $vehicleData['model'] ?? null,
                        ]);
                    }
                }
            }

            // Handle New Documents
            if ($request->has('documents')) {
                foreach ($request->documents as $doc) {
                    if (isset($doc['image'])) {
                        $path = $doc['image']->store('agents/documents', 'public');
                        $deliveryAgent->documents()->create([
                            'type' => $doc['type'],
                            'number' => $doc['number'] ?? null,
                            'image_path' => $path,
                        ]);
                    }
                }
            }

            // Handle New Vehicles
            if ($request->has('vehicles')) {
                foreach ($request->vehicles as $vehicle) {
                    $deliveryAgent->vehicles()->create([
                        'type' => $vehicle['type'],
                        'vehicle_number' => $vehicle['vehicle_number'],
                        'model' => $vehicle['model'] ?? null,
                    ]);
                }
            }

            $deliveryAgent->update($data);

            DB::commit();
            return redirect()->route('admin.delivery-agents.index')->with('success', 'Agent updated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Update failed: ' . $e->getMessage());
        }
    }

    public function toggleStatus(DeliveryAgent $deliveryAgent)
    {
        $deliveryAgent->status = !$deliveryAgent->status;
        $deliveryAgent->save();
        $deliveryAgent->user->update(['is_active' => $deliveryAgent->status]); // Sync user login status
        return back()->with('success', 'Status updated.');
    }

    public function destroy(DeliveryAgent $deliveryAgent)
    {
        // cleanup files
        if ($deliveryAgent->profile_image)
            Storage::disk('public')->delete($deliveryAgent->profile_image);

        foreach ($deliveryAgent->documents as $doc) {
            Storage::disk('public')->delete($doc->image_path);
        }

        $deliveryAgent->user->delete(); // Cascades delete to agent profile
        // $deliveryAgent->delete(); // Handled by cascade if configured, but explicit is fine or rely on DB FK

        return back()->with('success', 'Agent deleted.');
    }

    public function destroyDocument($id)
    {
        $doc = \App\Models\AgentDocument::findOrFail($id);
        if ($doc->image_path)
            Storage::disk('public')->delete($doc->image_path);

        $doc->delete();
        return back()->with('success', 'Document deleted.');
    }

    public function destroyVehicle($id)
    {
        $vehicle = \App\Models\AgentVehicle::findOrFail($id);
        $vehicle->delete();
        return back()->with('success', 'Vehicle deleted.');
    }

    public function showNotificationForm()
    {
        $agents = DeliveryAgent::with('user')->get();
        return view('admin.delivery_agents.notification', compact('agents'));
    }

    public function sendNotification(Request $request)
    {
        $request->validate([
            'agent_id' => 'required', // 'all' or agent id
            'title' => 'required|string|max:255',
            'message' => 'required|string',
        ]);

        $notification = new \App\Notifications\DeliveryAgentNotification($request->message, $request->title);

        if ($request->agent_id == 'all') {
            $agents = DeliveryAgent::with('user')->get();
            foreach ($agents as $agent) {
                if ($agent->user) {
                    $agent->user->notify($notification);
                }
            }
        } else {
            $agent = DeliveryAgent::with('user')->findOrFail($request->agent_id);
            if ($agent->user) {
                $agent->user->notify($notification);
            }
        }

        return redirect()->route('admin.delivery-agents.index')->with('success', 'Notification sent successfully.');
    }
}
