<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\CustomOffer;
use App\Models\CustomOfferItem;
use App\Models\Layout;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;

class CustomOfferController extends Controller
{
    public function index()
    {
        $offers = CustomOffer::with('layout', 'items')->latest()->get();
        return view('admin.settings.custom_offers.index', compact('offers'));
    }

    public function create()
    {
        $layouts = Layout::all();
        $pages = $this->getPlacementOptions();
        return view('admin.settings.custom_offers.create', compact('layouts', 'pages'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'layout_id' => 'required|exists:layouts,id',
            'banner_image.*' => 'required|image|max:10240', // Increased limit for upload
            'redirect_url.*' => 'nullable|url',
            'alt_tag.*' => 'nullable|string',
            'placement_pages' => 'nullable|array',
            'placement_pages.*' => 'string',
            'target_element' => 'nullable|string|max:255',
            'placement_position' => 'nullable|string|in:before,after,inside_start,inside_end',
            'border_radius' => 'nullable|string',
            'box_shadow' => 'nullable|string',
            'bg_color' => 'nullable|string',
            'heading_font_size' => 'nullable|string',
            'heading_font_weight' => 'nullable|string',
            'heading_color' => 'nullable|string',
        ]);

        $customOffer = CustomOffer::create([
            'name' => $request->name,
            'layout_id' => $request->layout_id,
            'status' => true,
            'placement_pages' => $request->placement_pages,
            'target_element' => $request->target_element,
            'placement_position' => $request->placement_position,
            'border_radius' => $request->border_radius,
            'box_shadow' => $request->box_shadow,
            'bg_color' => $request->bg_color,
            'heading_font_size' => $request->heading_font_size,
            'heading_font_weight' => $request->heading_font_weight,
            'heading_color' => $request->heading_color,
        ]);

        if ($request->hasFile('banner_image')) {
            foreach ($request->file('banner_image') as $index => $image) {
                // Process Image: Compress & WebP
                $path = $this->processImage($image);
                
                CustomOfferItem::create([
                    'custom_offer_id' => $customOffer->id,
                    'banner_image' => $path,
                    'redirect_url' => $request->redirect_url[$index] ?? null,
                    'alt_tag' => $request->alt_tag[$index] ?? null,
                    'sort_order' => $index,
                ]);
            }
        }

        return redirect()->route('admin.settings.custom_offers.index')->with('success', 'Custom Offer created successfully.');
    }

    public function edit(CustomOffer $customOffer)
    {
        $layouts = Layout::all();
        $pages = $this->getPlacementOptions();
        $customOffer->load('items');
        return view('admin.settings.custom_offers.edit', compact('customOffer', 'layouts', 'pages'));
    }

    private function getPlacementOptions()
    {
        // 1. Static Pages
        $options = collect([
            (object) ['id' => 'home', 'title' => 'Home Page', 'slug' => 'home'],
            (object) ['id' => 'shop', 'title' => 'Shop Page', 'slug' => 'shop']
        ]);

        // 2. CMS Pages
        $cmsPages = \App\Models\Page::select('id', 'title', 'slug')->get();
        foreach ($cmsPages as $page) {
            $options->push((object) [
                'id' => 'page-' . $page->id,
                'title' => $page->title . ' (Page)',
                'slug' => $page->slug
            ]);
        }

        // 3. Collection Pages (Subcategories: collections/*/*)
        $subcategories = \App\Models\Category::whereNotNull('parent_id')
            ->where('status', true)
            ->with('parent')
            ->get();

        foreach ($subcategories as $subcategory) {
            if ($subcategory->parent) {
                $slug = 'collections/' . $subcategory->parent->slug . '/' . $subcategory->slug;
                $options->push((object) [
                    'id' => 'cat-' . $subcategory->id,
                    'title' => $subcategory->parent->name . ' > ' . $subcategory->name . ' (Collection)',
                    'slug' => $slug
                ]);
            }
        }

        return $options;
    }

    public function update(Request $request, CustomOffer $customOffer)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'layout_id' => 'required|exists:layouts,id',
            'banner_image.*' => 'nullable|image|max:10240',
            'redirect_url.*' => 'nullable|url',
            'alt_tag.*' => 'nullable|string',
            'placement_pages' => 'nullable|array',
            'placement_pages.*' => 'string',
            'target_element' => 'nullable|string|max:255',
            'placement_position' => 'nullable|string|in:before,after,inside_start,inside_end',
            'border_radius' => 'nullable|string',
            'box_shadow' => 'nullable|string',
            'bg_color' => 'nullable|string',
            'heading_font_size' => 'nullable|string',
            'heading_font_weight' => 'nullable|string',
            'heading_color' => 'nullable|string',
        ]);

        // Helper to check if layout changed to warn or handle
        if ($customOffer->layout_id != $request->layout_id) {
            // If layout changes, we might need to delete old items or adjust them. 
        }

        $customOffer->update([
            'name' => $request->name,
            'status' => $request->has('status'), // Checkbox handling
            'placement_pages' => $request->placement_pages,
            'target_element' => $request->target_element,
            'placement_position' => $request->placement_position,
            'border_radius' => $request->border_radius,
            'box_shadow' => $request->box_shadow,
            'bg_color' => $request->bg_color,
            'heading_font_size' => $request->heading_font_size,
            'heading_font_weight' => $request->heading_font_weight,
            'heading_color' => $request->heading_color,
        ]);

        // Handle Layout Change: Delete old items ONLY if new images are provided or if we intentionaly want to reset
        if ($customOffer->layout_id != $request->layout_id) {
            // If layout changed, we MUST have new images for the new layout
            if (!$request->hasFile('banner_image')) {
                return back()->withErrors(['layout_id' => 'When changing the layout style, you must upload new banner images for the new layout columns.'])->withInput();
            }

             // Only update layout_id and delete old items if we're sure we have new ones
             $customOffer->update(['layout_id' => $request->layout_id]);
             
             // Refresh items relation to get current state
             $customOffer->load('items');
             
             // Delete old images
             foreach ($customOffer->items as $item) {
                 if ($item->banner_image && Storage::disk('public')->exists($item->banner_image)) {
                     Storage::disk('public')->delete($item->banner_image);
                 }
             }
             $customOffer->items()->delete();
        }
        
        // Handle Existing Items
        if ($request->has('item_id')) {
             $redirectUrls = $request->input('redirect_url', []);
             $altTags = $request->input('alt_tag', []);

             foreach ($request->item_id as $index => $itemId) {
                 $item = CustomOfferItem::find($itemId);
                 if ($item) {
                     $data = [
                         'redirect_url' => $redirectUrls[$index] ?? null,
                         'alt_tag' => $altTags[$index] ?? null,
                     ];
                     
                     if ($request->hasFile("banner_image.$index")) {
                         // Delete old image
                         if ($item->banner_image && Storage::disk('public')->exists($item->banner_image)) {
                             Storage::disk('public')->delete($item->banner_image);
                         }
                         // Process New Image
                         $data['banner_image'] = $this->processImage($request->file("banner_image.$index"));
                     }
                     
                     $item->update($data);
                 }
             }
        }
        
        // Handle New Items (if layout changed or added)
        if ($customOffer->wasChanged('layout_id')) {
             if ($request->hasFile('banner_image')) {
                $redirectUrls = $request->input('redirect_url', []);
                $altTags = $request->input('alt_tag', []);

                foreach ($request->file('banner_image') as $index => $image) {
                    $path = $this->processImage($image);
                    
                    CustomOfferItem::create([
                        'custom_offer_id' => $customOffer->id,
                        'banner_image' => $path,
                        'redirect_url' => $redirectUrls[$index] ?? null,
                        'alt_tag' => $altTags[$index] ?? null,
                        'sort_order' => $index,
                    ]);
                }
            }
        }

        return redirect()->route('admin.settings.custom_offers.index')->with('success', 'Custom Offer updated successfully.');
    }

    public function destroy(CustomOffer $customOffer)
    {
        foreach ($customOffer->items as $item) {
            if ($item->banner_image && Storage::disk('public')->exists($item->banner_image)) {
                Storage::disk('public')->delete($item->banner_image);
            }
        }
        $customOffer->delete();
        return back()->with('success', 'Custom Offer deleted successfully.');
    }

    /**
     * Process uploaded image: Compress and Convert to WebP
     */
    private function processImage($image)
    {
        $manager = new ImageManager(new Driver());
        
        $filename = Str::random(40) . '.webp';
        $path = 'custom_offers/' . $filename;

        // Ensure directory exists
        if (!Storage::disk('public')->exists('custom_offers')) {
            Storage::disk('public')->makeDirectory('custom_offers');
        }

        $img = $manager->read($image);

        // Compress and convert to WebP (Quality: 60)
        $encoded = $img->toWebp(60);
        
        Storage::disk('public')->put($path, $encoded);

        // Return path relative to public/storage (which storage link points to)
        // storage/custom_offers/filename.webp is how it's usually accessed if we use Storage::url($path)
        // If we just store relative path 'custom_offers/filename.webp', then Storage::url() adds /storage/ prefix.
        // In Store method we used $image->store() which returns relative path.
        // So we should return relative path here too.
        
        return $path;
    }
}
