<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;

class ClientController extends Controller
{
    public function index(Request $request)
    {
        $query = User::doesntHave('roles')->withCount('orders');

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->filled('sort')) {
            $direction = $request->get('direction', 'asc') === 'desc' ? 'desc' : 'asc';

            switch ($request->sort) {
                case 'name':
                    $query->orderBy('name', $direction);
                    break;
                case 'orders_count':
                    $query->orderBy('orders_count', $direction);
                    break;
                case 'wallet_balance':
                    $query->orderBy('wallet_balance', $direction);
                    break;
                default:
                    $query->latest();
                    break;
            }
        } else {
            $query->latest();
        }

        $clients = $query->paginate(10)->withQueryString();
        return view('admin.clients.index', compact('clients'));
    }

    public function show(User $client)
    {
        if ($client->roles()->exists()) {
            return redirect()->route('admin.users.show', $client);
        }

        $client->loadCount('orders');
        // Load recent orders for the dashboard-like view
        $recentOrders = $client->orders()->latest()->take(5)->get();

        return view('admin.clients.show', compact('client', 'recentOrders'));
    }

    public function orders(User $client)
    {
        if ($client->roles()->exists()) {
            return redirect()->route('admin.users.idx', $client);
        }

        $orders = $client->orders()->latest()->paginate(10);
        return view('admin.clients.orders', compact('client', 'orders'));
    }

    public function sendNotification(Request $request, User $client, \App\Services\FCMService $fcmService)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'required|string',
        ]);

        if (!$client->fcm_token) {
            return back()->with('error', 'Client does not have an FCM token.');
        }

        try {
            $fcmService->sendNotification(
                $client->fcm_token,
                $request->title,
                $request->body,
                ['type' => 'admin_message']
            );

            return back()->with('success', 'Notification sent successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to send notification: ' . $e->getMessage());
        }
    }
}
