<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ChatbotSetting;
use App\Models\ChatbotKnowledgeBase;

class ChatbotController extends Controller
{
    public function index()
    {
        $chatbotSettings = ChatbotSetting::find(1);

        if (!$chatbotSettings) {
            $chatbotSettings = ChatbotSetting::create([
                'id' => 1,
                'name' => 'Support Bot',
                'color_scheme' => '#9B2D69',
                'welcome_message' => 'Hello! How can I help you today?',
                'is_active' => true,
            ]);
        }

        $knowledgeBase = ChatbotKnowledgeBase::latest()->get();

        return view('admin.chatbot.index', compact('chatbotSettings', 'knowledgeBase'));
    }

    public function updateSettings(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'color_scheme' => 'required|string|max:7',
            'welcome_message' => 'nullable|string',
            'system_prompt' => 'nullable|string',
        ]);

        $chatbotSettings = ChatbotSetting::find(1);
        if (!$chatbotSettings) {
            $chatbotSettings = new ChatbotSetting();
            $chatbotSettings->id = 1;
        }

        $chatbotSettings->name = $request->name;
        $chatbotSettings->color_scheme = $request->color_scheme;
        $chatbotSettings->is_active = $request->has('is_active');
        $chatbotSettings->welcome_message = $request->welcome_message;
        $chatbotSettings->system_prompt = $request->system_prompt;
        $chatbotSettings->save();

        return redirect()->back()->with('success', 'Chatbot settings updated successfully.');
    }

    public function storeTrainingData(Request $request)
    {
        $request->validate([
            'question' => 'required|string',
            'keywords' => 'nullable|string',
            'answer' => 'required|string',
            'topic' => 'nullable|string',
        ]);

        ChatbotKnowledgeBase::create([
            'question' => $request->question,
            'keywords' => $request->keywords,
            'answer' => $request->answer,
            'topic' => $request->topic,
            'type' => 'manual',
        ]);

        return redirect()->back()->with('success', 'Knowledge base entry added.');
    }

    public function destroyTrainingData($id)
    {
        ChatbotKnowledgeBase::findOrFail($id)->delete();
        return redirect()->back()->with('success', 'Entry deleted.');
    }

    public function learn()
    {
        // 1. Clean up old auto-generated entries
        ChatbotKnowledgeBase::where('type', 'learned')->delete();

        // 2. System-defined functional Q&A (Simulate scraping functionalities)
        $functionalities = [
            [
                'topic' => 'Order Placement',
                'question' => 'How to place an order?',
                'answer' => "To place an order: 1. Browse our products and click 'Add to Cart'. 2. Go to your Cart and click 'Checkout'. 3. Enter your shipping details and choose a payment method. 4. Confirm your order. You will receive an email confirmation.",
            ],
            [
                'topic' => 'Order Cancellation',
                'question' => 'How to cancel an order?',
                'answer' => "You can cancel your order from the 'My Orders' section in your profile if it hasn't been shipped yet. If it has been shipped, please reach out to our support team for a return request.",
            ],
            [
                'topic' => 'Order Tracking',
                'question' => 'How to track my order?',
                'answer' => "Go to 'My Orders' in your profile. Find your order and click 'Track Order'. You will see the current status and tracking number.",
            ],
            [
                'topic' => 'Invoice',
                'question' => 'How to get an invoice?',
                'answer' => "You can download your invoice from the 'My Orders' section. Click on the order details, and look for the 'Download Invoice' button.",
            ],
            [
                'topic' => 'Reviews',
                'question' => 'How to give reviews?',
                'answer' => "Go to the product page of the item you purchased. Scroll down to the 'Reviews' section and click 'Write a Review'. You can rate functionality and add photos.",
            ],
            [
                'topic' => 'Wallet System',
                'question' => 'How does the wallet system work?',
                'answer' => "Your wallet stores credits from refunds or gift cards. You can use your wallet balance to pay for orders partially or fully during checkout.",
            ],
            [
                'topic' => 'Profile Update',
                'question' => 'How to update profile?',
                'answer' => "Go to 'My Profile' in the dashboard. You can update your name, email, and profile picture there. Click 'Save Changes' to update.",
            ],
            [
                'topic' => 'Address Management',
                'question' => 'How to add addresses?',
                'answer' => "Go to 'My Addresses' in your dashboard. Click 'Add New Address', fill in the details, and save. You can manage multiple addresses.",
            ],
            [
                'topic' => 'Support Ticket',
                'question' => 'How to raise a ticket?',
                'answer' => "Go to 'My Tickets' or 'Help & Support'. Click 'Create Ticket', select a subject, describe your issue, and submit. Our support team will respond shortly.",
            ],
            [
                'topic' => 'Redeem Points',
                'question' => 'How to redeem points?',
                'answer' => "If you have loyalty points, you can redeem them on the cart page. Look for the 'Redeem Points' option before checkout to get a discount."
            ],
            [
                'topic' => 'Shipping Policy',
                'question' => 'What is the shipping policy?',
                'answer' => "Our shipping policy outlines delivery times and costs. Check the 'Shipping Policy' page in the footer for details.",
            ],
            [
                'topic' => 'Cancellation Policy',
                'question' => 'What is the cancellation policy?',
                'answer' => "You can cancel orders before shipment. Refund will be processed to your wallet or original payment method. See 'Cancellation Policy' page.",
            ],
            [
                'topic' => 'Coupons',
                'question' => 'How to apply a coupon?',
                'answer' => "On the Cart or Checkout page, enter your coupon code in the 'Apply Coupon' box and click 'Apply'. The discount will be reflected in the total.",
            ],
            [
                'topic' => 'Gift Cards',
                'question' => 'How to create gift cards?',
                'answer' => "Go to 'My Gift Cards' in the dashboard. Click 'Create Gift Card', select amount and recipient, and proceed to payment. The recipient will receive the card via email.",
            ],
            [
                'topic' => 'Ticket Reply',
                'question' => 'How to reply to a ticket?',
                'answer' => "Go to 'My Tickets', open the ticket you want to reply to, type your message in the reply box, and click 'Send Reply'.",
            ],
        ];

        foreach ($functionalities as $func) {
            ChatbotKnowledgeBase::create([
                'topic' => $func['topic'],
                'question' => $func['question'],
                'answer' => $func['answer'],
                'type' => 'learned'
            ]);
        }

        // 3. Add Contact Info from Settings
        $settings = \App\Models\GeneralSetting::first();
        if ($settings) {
            // Address
            $address = $settings->address_head ?? ($settings->contact_address ?? null);
            if ($address) {
                ChatbotKnowledgeBase::create([
                    'topic' => 'Location',
                    'question' => 'Where are you located?',
                    'answer' => "We are located at: " . $address,
                    'type' => 'learned'
                ]);
            }

            // Phone
            $phone = $settings->phone_sales ?? ($settings->phone_support ?? null);
            if ($phone) {
                ChatbotKnowledgeBase::create([
                    'topic' => 'Contact',
                    'question' => 'What is your phone number?',
                    'answer' => "You can reach us at: " . $phone,
                    'type' => 'learned'
                ]);
            }

            // Email
            $email = $settings->email_sales ?? ($settings->email_support ?? null);
            if ($email) {
                ChatbotKnowledgeBase::create([
                    'topic' => 'Contact',
                    'question' => 'What is your email?',
                    'answer' => "You can email us at: " . $email,
                    'type' => 'learned'
                ]);
            }
        }

        // 4. Add Product Info
        $products = \App\Models\Product::where('status', true)->get();
        foreach ($products as $product) {
            // Use generic route or assume named route exists
            $link = route('shop.product', $product->slug);
            $price = $product->sale_price ?? $product->price;
            $desc = \Illuminate\Support\Str::limit(strip_tags($product->short_description ?? $product->description), 150);

            // General Info
            ChatbotKnowledgeBase::create([
                'topic' => 'Product: ' . $product->name,
                'question' => 'Tell me about ' . $product->name,
                'answer' => "{$product->name} is available for checkout. Price: {$price}. {$desc}. View details: {$link}",
                'type' => 'learned'
            ]);

            // Availability
            ChatbotKnowledgeBase::create([
                'topic' => 'Product Availability',
                'question' => 'Do you have ' . $product->name . '?',
                'answer' => "Yes, we have {$product->name} in stock! You can buy it here: {$link}",
                'type' => 'learned'
            ]);
        }

        return redirect()->back()->with('success', 'Bot has learned from system data, contact info, and ' . $products->count() . ' products.');
    }
}
