<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CategoryController extends Controller
{
    public function index(Request $request)
    {
        $query = Category::with('parent')->latest();

        if ($request->has('search')) {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%")
                ->orWhere('slug', 'like', "%{$search}%");
        }

        $categories = $query->paginate(20);
        return view('admin.categories.index', compact('categories'));
    }

    public function toggleStatus(Category $category)
    {
        $category->status = !$category->status;
        $category->save();

        return response()->json(['success' => true, 'status' => $category->status, 'message' => 'Status updated successfully']);
    }

    public function create()
    {
        $categories = Category::whereNull('parent_id')->get();
        return view('admin.categories.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'slug' => 'nullable|alpha_dash|unique:categories,slug',
            'image' => 'nullable|image|max:5120', // Max 5MB
            'parent_id' => 'nullable|exists:categories,id',
            'status' => 'boolean',
            'schema_markup' => 'nullable|string',
            'seo_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'breadcrumbs_title' => 'nullable|string|max:255',
            'canonical_url' => 'nullable|url',
            'is_nofollow' => 'boolean',
        ]);

        $slug = $request->slug ? Str::slug($request->slug) : Str::slug($request->name);

        // Ensure slug is unique if it was auto-generated (or even if manual, to be safe, though validation covers unique)
        if (!$request->slug && Category::where('slug', $slug)->exists()) {
            $slug = $slug . '-' . time();
        }

        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = '/storage/' . $request->file('image')->store('categories', 'public');
        }

        Category::create([
            'name' => $request->name,
            'slug' => $slug,
            'image_url' => $imagePath,
            'parent_id' => $request->parent_id,
            'status' => $request->status ? true : false,
            'schema_markup' => $request->schema_markup,
            'seo_title' => $request->seo_title,
            'meta_description' => $request->meta_description,
            'breadcrumbs_title' => $request->breadcrumbs_title,
            'canonical_url' => $request->canonical_url,
            'is_nofollow' => $request->is_nofollow ? true : false,
        ]);

        return redirect()->route('admin.categories.index')->with('success', 'Category created successfully');
    }

    public function edit(Category $category)
    {
        $categories = Category::whereNull('parent_id')->where('id', '!=', $category->id)->get();
        return view('admin.categories.edit', compact('category', 'categories'));
    }

    public function update(Request $request, Category $category)
    {
        $request->validate([
            'name' => 'required',
            'slug' => 'nullable|alpha_dash|unique:categories,slug,' . $category->id,
            'image' => 'nullable|image|max:5120',
            'parent_id' => 'nullable|exists:categories,id',
            'status' => 'boolean',
            'schema_markup' => 'nullable|string',
            'seo_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'breadcrumbs_title' => 'nullable|string|max:255',
            'canonical_url' => 'nullable|url',
            'is_nofollow' => 'boolean',
        ]);

        $slug = $category->slug; // Default to current slug

        if ($request->filled('slug')) {
            $slug = Str::slug($request->slug);
        } elseif ($request->name !== $category->name) {
            // If name changed but slug input empty (shouldn't happen if we populate it), 
            // but if they cleared it, maybe they want auto-gen?
            $slug = Str::slug($request->name);
            if (Category::where('slug', $slug)->where('id', '!=', $category->id)->exists()) {
                $slug = $slug . '-' . time();
            }
        }

        $data = [
            'name' => $request->name,
            'slug' => $slug,
            'parent_id' => $request->parent_id,
            'status' => $request->status ? true : false,
            'schema_markup' => $request->schema_markup,
            'seo_title' => $request->seo_title,
            'meta_description' => $request->meta_description,
            'breadcrumbs_title' => $request->breadcrumbs_title,
            'canonical_url' => $request->canonical_url,
            'is_nofollow' => $request->is_nofollow ? true : false,
        ];

        if ($request->hasFile('image')) {
            // Delete old image if needed (optional optimization)
            $data['image_url'] = '/storage/' . $request->file('image')->store('categories', 'public');
        }

        $category->update($data);

        return redirect()->route('admin.categories.index')->with('success', 'Category updated successfully');
    }

    public function destroy(Category $category)
    {
        $category->delete();
        return redirect()->route('admin.categories.index')->with('success', 'Category deleted successfully');
    }
}
