<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\EmailTemplate;
use Illuminate\Support\Facades\Mail;
use App\Mail\BulkEmail;
use Maatwebsite\Excel\Facades\Excel;

use App\Imports\BulkEmailImport;

class BulkEmailController extends Controller
{
    public function index()
    {
        $users = User::select('id', 'name', 'email')->get();
        $templates = EmailTemplate::where('is_active', 1)->select('id', 'name', 'subject')->get();
        $campaigns = \App\Models\BulkEmailCampaign::latest()->paginate(10);
        return view('admin.bulk-email.index', compact('users', 'templates', 'campaigns'));
    }

    public function send(Request $request)
    {
        $request->validate([
            'recipient_type' => 'required|in:all,single,excel',
            'user_id' => 'required_if:recipient_type,single|exists:users,id',
            'file' => 'required_if:recipient_type,excel|file|mimes:xlsx,xls,csv',
            'template_id' => 'required|exists:email_templates,id',
            'subject' => 'required|string|max:255',
        ]);

        $recipientType = $request->recipient_type;
        $templateId = $request->template_id;
        $subject = $request->subject;

        $template = EmailTemplate::findOrFail($templateId);
        $content = $template->content;

        $recipients = [];

        if ($recipientType === 'all') {
            $recipients = User::where('is_active', true)->get(); // Get Models
        } elseif ($recipientType === 'single') {
            $user = User::find($request->user_id);
            if ($user && $user->email) {
                $recipients = collect([$user]);
            } else {
                $recipients = collect([]);
            }
        } elseif ($recipientType === 'excel') {
            // ... existing excel logic ...
            // Excel logic stays as strings for now, unless we want to lookup users
            if ($request->hasFile('file')) {
                // ... (keep excel parsing logic, store in $excelEmails) ...
                // Refactoring excel logic slightly to be cleaner
                $excelEmails = [];
                try {
                    $rows = Excel::toArray(new BulkEmailImport, $request->file('file'));
                    if (count($rows) > 0) {
                        $sheet = $rows[0];
                        $header = $sheet[0] ?? [];
                        $emailIndex = -1;
                        foreach ($header as $index => $col) {
                            if (strtolower(trim($col)) === 'email') {
                                $emailIndex = $index;
                                break;
                            }
                        }
                        if ($emailIndex === -1)
                            $emailIndex = 0;
                        $startIndex = ($emailIndex === -1 || strtolower(trim($header[$emailIndex])) === 'email') ? 1 : 0;
                        for ($i = $startIndex; $i < count($sheet); $i++) {
                            $row = $sheet[$i];
                            if (isset($row[$emailIndex]) && filter_var($row[$emailIndex], FILTER_VALIDATE_EMAIL)) {
                                $excelEmails[] = $row[$emailIndex];
                            }
                        }
                    }
                } catch (\Exception $e) {
                    return back()->withErrors(['file' => 'Error reading file: ' . $e->getMessage()]);
                }
                $recipients = collect($excelEmails); // These are strings
            }
        }

        if ($recipients->isEmpty()) {
            return back()->withErrors(['No valid recipients found.']);
        }

        $count = $recipients->count(); // Approximate

        // Save Campaign Log
        $campaign = \App\Models\BulkEmailCampaign::create([
            'subject' => $subject, // Original subject with placeholders
            'recipient_type' => $recipientType,
            'sent_count' => $count,
            'status' => 'sent',
            'template_id' => $template->id,
            'template_name' => $template->name,
        ]);

        foreach ($recipients as $recipient) {
            // Detect if User Model or Email String
            $user = ($recipient instanceof User) ? $recipient : User::where('email', $recipient)->first();
            $email = ($recipient instanceof User) ? $recipient->email : $recipient;

            // Prepare Data for Replacement
            $data = [
                'customer_name' => $user ? $user->name : 'Customer',
                'app_name' => config('app.name'),
                'email' => $email
            ];

            // Special Logic for Reminders
            if ($template->category == 'Cart Reminder' && $user) {
                $cart = \App\Models\Cart::where('user_id', $user->id)->with('items.product')->first();
                $data['cart_items_summary'] = \App\Services\UserMailService::getCartItemsSummary($cart);
                $data['cart_url'] = route('cart.index');
            } elseif ($template->category == 'Wishlist Reminder' && $user) {
                $wishlistItems = \App\Models\Wishlist::where('user_id', $user->id)->with('product')->get();
                $data['wishlist_items_summary'] = \App\Services\UserMailService::getWishlistItemsSummary($wishlistItems);
                $data['wishlist_url'] = route('user.wishlist');
            } else {
                // Defaults to avoid undefined errors if template has them
                $data['cart_items_summary'] = '';
                $data['cart_url'] = url('/');
                $data['wishlist_items_summary'] = '';
                $data['wishlist_url'] = url('/');
            }

            // Parse Content & Subject
            $userContent = $content;
            $userSubject = $subject;

            foreach ($data as $key => $val) {
                // Ensure value is string
                if (!is_string($val))
                    $val = (string) $val;
                $userContent = str_replace('{{ ' . $key . ' }}', $val, $userContent);
                $userSubject = str_replace('{{ ' . $key . ' }}', $val, $userSubject);
                // Also support without spaces
                $userContent = str_replace('{{' . $key . '}}', $val, $userContent);
                $userSubject = str_replace('{{' . $key . '}}', $val, $userSubject);
            }

            try {
                Mail::to($email)->send(new BulkEmail($userSubject, $userContent));

                \App\Models\BulkEmailLog::create([
                    'bulk_email_campaign_id' => $campaign->id,
                    'recipient_email' => $email,
                    'status' => 'sent',
                    'sent_at' => now(),
                ]);
            } catch (\Exception $e) {
                \Log::error("Failed to send bulk email to $email: " . $e->getMessage());
                \App\Models\BulkEmailLog::create([
                    'bulk_email_campaign_id' => $campaign->id,
                    'recipient_email' => $email,
                    'status' => 'failed',
                    'error_details' => $e->getMessage(),
                ]);
            }
        }

        return redirect()->route('admin.bulk-email.index')->with('success', "Emails sent/queued to $count recipients.");
    }

    public function showLogs($id)
    {
        $campaign = \App\Models\BulkEmailCampaign::findOrFail($id);
        $logs = $campaign->logs()->latest()->paginate(20);
        return view('admin.bulk-email.logs', compact('campaign', 'logs'));
    }

    public function resend($id)
    {
        $campaign = \App\Models\BulkEmailCampaign::findOrFail($id);

        // Get unique emails from logs
        $recipients = $campaign->logs()->pluck('recipient_email')->unique();

        $template = EmailTemplate::find($campaign->template_id);

        if (!$template) {
            return back()->withErrors(['Template for this campaign no longer exists.']);
        }

        $content = $template->content;
        $subject = $campaign->subject;

        foreach ($recipients as $email) {
            // Lookup user for dynamic data
            $user = User::where('email', $email)->first();

            // Prepare Data for Replacement
            $data = [
                'customer_name' => $user ? $user->name : 'Customer',
                'app_name' => config('app.name'),
                'email' => $email
            ];

            // Special Logic for Reminders
            if ($template->category == 'Cart Reminder' && $user) {
                $cart = \App\Models\Cart::where('user_id', $user->id)->with('items.product')->first();
                $data['cart_items_summary'] = \App\Services\UserMailService::getCartItemsSummary($cart);
                $data['cart_url'] = route('cart.index');
            } elseif ($template->category == 'Wishlist Reminder' && $user) {
                $wishlistItems = \App\Models\Wishlist::where('user_id', $user->id)->with('product')->get();
                $data['wishlist_items_summary'] = \App\Services\UserMailService::getWishlistItemsSummary($wishlistItems);
                $data['wishlist_url'] = route('user.wishlist');
            } else {
                // Defaults
                $data['cart_items_summary'] = '';
                $data['cart_url'] = url('/');
                $data['wishlist_items_summary'] = '';
                $data['wishlist_url'] = url('/');
            }

            // Parse Content & Subject
            $userContent = $content;
            $userSubject = $subject;

            foreach ($data as $key => $val) {
                if (!is_string($val))
                    $val = (string) $val;
                $userContent = str_replace('{{ ' . $key . ' }}', $val, $userContent);
                $userSubject = str_replace('{{ ' . $key . ' }}', $val, $userSubject);
                $userContent = str_replace('{{' . $key . '}}', $val, $userContent);
                $userSubject = str_replace('{{' . $key . '}}', $val, $userSubject);
            }

            try {
                Mail::to($email)->send(new BulkEmail($userSubject, $userContent));

                \App\Models\BulkEmailLog::create([
                    'bulk_email_campaign_id' => $campaign->id,
                    'recipient_email' => $email,
                    'status' => 'sent',
                    'sent_at' => now(),
                ]);

            } catch (\Exception $e) {
                \Log::error("Failed to resend bulk email to $email: " . $e->getMessage());
                \App\Models\BulkEmailLog::create([
                    'bulk_email_campaign_id' => $campaign->id,
                    'recipient_email' => $email,
                    'status' => 'failed',
                    'error_details' => $e->getMessage(),
                ]);
            }
        }

        return back()->with('success', 'Campaign emails resent successfully.');
    }

    public function destroy($id)
    {
        $campaign = \App\Models\BulkEmailCampaign::findOrFail($id);
        $campaign->delete();
        return redirect()->route('admin.bulk-email.index')->with('success', 'Campaign deleted successfully.');
    }
}
