<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class BlogController extends Controller
{
    public function index()
    {
        $blogs = \App\Models\BlogPost::latest()->paginate(100);
        return view('admin.blogs.index', compact('blogs'));
    }

    public function create()
    {
        return view('admin.blogs.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'status' => 'required|in:draft,published',
            'published_at' => 'nullable|date',
            'excerpt' => 'nullable|string|max:500',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'schema_markup' => 'nullable|string',
        ]);

        $data = $request->except('image');
        $data['slug'] = \Illuminate\Support\Str::slug($request->title);

        // Ensure slug is unique
        if (\App\Models\BlogPost::where('slug', $data['slug'])->exists()) {
            $data['slug'] = $data['slug'] . '-' . time();
        }

        if ($request->hasFile('image')) {
            $data['image'] = $request->file('image')->store('blog', 'public');
        }

        if ($request->status == 'published' && !$request->published_at) {
            $data['published_at'] = now();
        }

        \App\Models\BlogPost::create($data);

        return redirect()->route('admin.blogs.index')->with('success', 'Blog post created successfully.');
    }

    public function edit($id)
    {
        $blog = \App\Models\BlogPost::findOrFail($id);
        return view('admin.blogs.edit', compact('blog'));
    }

    public function update(Request $request, $id)
    {
        $blog = \App\Models\BlogPost::findOrFail($id);

        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'status' => 'required|in:draft,published',
            'published_at' => 'nullable|date',
            'excerpt' => 'nullable|string|max:500',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'schema_markup' => 'nullable|string',
        ]);

        $data = $request->except('image');

        if ($blog->title !== $request->title) {
            $data['slug'] = \Illuminate\Support\Str::slug($request->title);
            if (\App\Models\BlogPost::where('slug', $data['slug'])->where('id', '!=', $id)->exists()) {
                $data['slug'] = $data['slug'] . '-' . time();
            }
        }

        if ($request->hasFile('image')) {
            if ($blog->image) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($blog->image);
            }
            $data['image'] = $request->file('image')->store('blog', 'public');
        }

        $blog->update($data);

        return redirect()->route('admin.blogs.index')->with('success', 'Blog post updated successfully.');
    }

    public function destroy($id)
    {
        $blog = \App\Models\BlogPost::findOrFail($id);
        if ($blog->image) {
            \Illuminate\Support\Facades\Storage::disk('public')->delete($blog->image);
        }
        $blog->delete();
        return redirect()->route('admin.blogs.index')->with('success', 'Blog post deleted successfully.');
    }

    public function toggleStatus($id)
    {
        $blog = \App\Models\BlogPost::findOrFail($id);
        $blog->status = $blog->status === 'published' ? 'draft' : 'published';

        if ($blog->status === 'published' && !$blog->published_at) {
            $blog->published_at = now();
        }

        $blog->save();

        return back()->with('success', 'Blog status updated successfully.');
    }
}
