<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\UserActivity;
use Illuminate\Support\Facades\DB;

class AnalyticsController extends Controller
{
    public function funnel(Request $request)
    {
        // 1. Visitors (Unique Sessions)
        $totalVisitors = UserActivity::distinct('session_id')->count('session_id');

        // 2. Interest (Viewed any product)
        $interest = UserActivity::where('event_type', 'view_product')
            ->distinct('session_id')
            ->count('session_id');

        // 3. Desire (Added to Cart)
        $desire = UserActivity::where('event_type', 'add_to_cart')
            ->distinct('session_id')
            ->count('session_id');

        // 4. Action (Initiated Checkout)
        // Can be either 'initiate_checkout' event or having visited checkout page
        $action = UserActivity::where(function ($q) {
            $q->where('event_type', 'initiate_checkout')
                ->orWhere('url', 'like', '%/checkout%');
        })
            ->distinct('session_id')
            ->count('session_id');

        // 5. Conversion (Purchase)
        $conversion = UserActivity::where('event_type', 'purchase')
            ->distinct('session_id')
            ->count('session_id');

        // Calculate Drop-offs
        $funnelData = [
            'visitors' => $totalVisitors,
            'interest' => $interest,
            'desire' => $desire,
            'action' => $action,
            'conversion' => $conversion,
        ];

        // Ensure steps don't exceed previous step (logic correction for simple funnel)
        // In real world, users skip steps, but for a funnel view, we assume linear progression or raw counts.
        // Raw counts are safer.

        // Recent Sessions for Replay
        $recentSessions = UserActivity::select('session_id', 'user_id', 'ip_address', DB::raw('MAX(created_at) as last_active'))
            ->groupBy('session_id', 'user_id', 'ip_address')
            ->orderBy('last_active', 'desc')
            ->limit(20)
            ->with('user')
            ->get();

        return view('admin.analytics.funnel', compact('funnelData', 'recentSessions'));
    }

    public function viewSession($sessionId)
    {
        $activities = UserActivity::where('session_id', $sessionId)
            ->orderBy('created_at', 'asc')
            ->get();

        return view('admin.analytics.session_view', compact('activities', 'sessionId'));
    }
}
